/*
*******************************************************************************
*
* File:         dynamic.c
* Description:  Dynamic Capabilities setup, delete and query routines
* Author:       Tilmann Wendel, BID PL24 R&D
* Created:      Sep 9, 1997
* Modified:     Fri May 12 17:07:02 2000 by Henrik Liebau, BVS R&D
* Language:     C
* Package:      E2925A C-API
* Status:       Experimental (Do Not Distribute)
*
* (C) Copyright 1998, Hewlett-Packard GmbH, all rights reserved.
*
*******************************************************************************
*/

#if defined(_MSC_VER)
/* #pragma data_seg("dyn_data","FAR_DATA") */
/*** Not used; MSVC 1.52 bug (see Q128553) ***/
# pragma code_seg("dyn_code1","CODE")
#elif defined(__BORLANDC__)
# pragma option -dc
# pragma warn -use
# pragma warn -cln
#endif /* _MSC_VER etc. */


#include <typedefs.h>

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <ladata.h>
#include <regx10.h>
#include <session.h>

#include <b_streq.h>

/* ** Segmentation for DOS **
* MS allows us to put const data into its own named segment (data_seg()).
* Borland doesn't like that but does allow us to pack strings
* into the code segment (option -dc).
* Borland also ignores unknown pragmas; MS compilers give you a warning
*/

/* names.h contains the stringlists that are generated from the
   *defs.h files in ../include by parsnames.cl */
#include "names.h"

#define DO_STANDARD_ERROR_HANDLING           1
#define DO_NO_ERROR_HANDLING                 0


#define PROPERTY_ENTRY	  NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL


/* ------------------------------------------------------------------------
 * Data fields
 * Note: Leave the NULL at the end of the Stringlists. The compiler needs
 * it.
 * ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------
 *  Xact Compiler String Lists follow here
 * ------------------------------------------------------------------------ */
/* extern const b_param_stringlisttype pci_cmd_table[]; */
extern const b_param_stringlisttype byten_table[];
/* extern const b_param_stringlisttype term_table[]; */
/* extern const b_param_stringlisttype onoff_table[]; */
/* extern const b_param_stringlisttype toggle_table[]; */

/* this feature is not supported by the Ichiban any longer (due to timing) */
/* We still accept waitmode for compatibility reasons, but the hardware */
/* ignores this value. Make this active if the second cut can do it. */
/* extern const b_param_stringlisttype stepmode_table[]; */
/* extern const b_param_stringlisttype lock_table[]; */
extern const b_param_stringlisttype yesno_table[];
/* extern const b_param_stringlisttype waits_table[]; */
/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------
 * Some things are solved using Tables for the different Hardwares
 * These things are signals for trace memory and pattern terms.
 * ------------------------------------------------------------------------ */

/* the layout of the signaltable is as follows: (squeeze it into b_paraminfo)
 * new:  name, nameshort (always NULL), startbitpos (in min_val),
 * width (in max_val), default, NULL, shiftvalue (in paramrule).
 * the remaining infos within the struct are not entered here, the compiler
 * just leaves them uninitialized, but we do not need them
 * the official table in the analyzer class is built there */
/* every list needs a terminating line (important !! ) */

extern b_param_infotype e2925a_signal_table[];
extern b_param_infotype e2925a_deep_signal_table[];
extern b_param_infotype e2926a_signal_table[];
extern b_param_infotype e2928a_signal_table[];
extern b_param_infotype e2925b_signal_table[];
extern b_param_infotype e2926a_tc_signal_table[];
extern b_param_infotype e2925b_tc_signal_table[];

#define FIRMWARE_UX_COMPILE
#include "pattdefs.h"
#undef FIRMWARE_UX_COMPILE

extern b_param_infotype e2925a_patt_signal_table[];
extern b_param_infotype e2925a_deep_patt_signal_table[];


/* --------------------------------------------------------------------------
 * We use malloc so many times in here that this is worth it.
 * -------------------------------------------------------------------------- */
static b_errtype malloc_chk_zero(void **ptr, size_t size)
{
  if (NULL == (*ptr = malloc(size)))
  {
    return (B_E_HOST_MEM_FULL);
  }

  memset(*ptr, 0, size);
  return (B_E_OK);
}


/* ------------------------------------------------------------------------
 * This fct reads the capabilities out of the hardware
 * ------------------------------------------------------------------------ */
b_errtype BestIHWCapabilityGet(b_handletype handle, b_int32 * lic)
{
  b_errtype err;
  b_int8 buf[12];
  b_int16 buflen = OUT_LICENSE_GET;
  b_int64 temp;

  /* determine the license here.
   * for the old e2925 and e2925 with deep memory the license is fixed and
   * set to normal CAPI Access (see below). 
   * The Ichiban boards get the license from the hardware directly 
   */

  err = B_E_OK;

  if (handle_array[handle].port == B_PORT_OFFLINE)
  {
    temp = (b_int64) handle_array[handle].portnumber & ~B_CAPABILITY_HARDWARE_MASK;
    *lic = (b_int32) temp; 			
  }
  else if (BestIs2925(handle))
  {
    *lic = B_CAPABILITY_CAPI | B_CAPABILITY_ANALYZER | B_CAPABILITY_EXERCISER |
      B_CAPABILITY_HOSTINT | B_CAPABILITY_SST | B_CAPABILITY_PPR;
  }
  else if (BestHasIchiban(handle))
  {
    err = BestBasicCommand(handle, CMD_LICENSE_GET, NULL, IN_LICENSE_GET,
      buf, &buflen);
    (void) BestStream2Long(lic, buf, 3UL);
  }
  else
    *lic = 0;

  B_ERRETURN(err);
}
/* ------------------------------------------------------------------------
 * This fct writes the license back to the hardware
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestIHWCapabilitySet(b_handletype handle, b_int32 * lic)
{
  b_errtype err;
  b_int8 buf[12];

  if ((handle_array[handle].port == B_PORT_OFFLINE) || 
      !BestHasIchiban(handle))
  {
    err = B_E_WRONG_HW;
  }
  else
  {
    (void) BestLong2Stream(buf, lic, 3UL);
    err = BestBasicCommand(handle, CMD_LICENSE_SET, buf, IN_LICENSE_SET,
      NULL, NULL);
  }

  B_ERRETURN(err);
}


/* ------------------------------------------------------------------------
 * This function adds  a new parameter to the list of already available
 * parameters. In case of the first access, it starts with a completely new
 * list of parameters.
 * ------------------------------------------------------------------------ */
b_errtype BestAddParam(
    b_param_infotype ** anchor, /* add the data to this list */
    size_t num_elems,          /* how many elements do we have without the
                                 * new one */
    b_charptrtype param_name,   /* namestring of standard form */
    b_charptrtype param_short,  /* namestring of the short form */
    b_int32 min_val,            /* maximum value also bitstart(signal) */
    b_int32 max_val,            /* minimum value also bitend (signal) */
    b_int32 defaultval,         /* default value of this parameter */
    b_charptrtype defaultstr,   /* default string of this parameter */
    b_int32 paramrule,          /* param might have a special rule */
    b_int32 proptyp,            /* property for the C-API call (union) */
    const b_param_stringlisttype * stringlist,  /* array of possible
                                                 * stringlists */
    b_syntaxParamType syntax_m_block, /* compiler syntax information */
    b_syntaxParamType syntax_m_xact,
    b_syntaxParamType syntax_m_data,
    b_syntaxParamType syntax_m_last,
    b_syntaxParamType syntax_m_attr,
    b_syntaxParamType syntax_t_attr,
    b_int32 grouptyp            /* what group the property belongs to */
)
{
  b_param_infotype *pLastAnchor;
  *anchor = (b_param_infotype *) realloc((void *) (*anchor),
    (size_t) ((num_elems + 1) * sizeof(b_param_infotype)));
  if (*anchor == NULL)
    return (B_E_HOST_MEM_FULL); /* could not allocate memory */

  pLastAnchor = &((*anchor)[(size_t) num_elems]);

  pLastAnchor->param_name = param_name;
  pLastAnchor->param_short = param_short;
  pLastAnchor->min_val = min_val;
  pLastAnchor->max_val = max_val;
  pLastAnchor->defaultval = defaultval;
  pLastAnchor->defaultstr = defaultstr;
  pLastAnchor->paramrule = paramrule;
  pLastAnchor->proptyp.value = proptyp;
  pLastAnchor->stringlist = stringlist;
  pLastAnchor->syntaxinfo[0] = syntax_m_block;
  pLastAnchor->syntaxinfo[1] = syntax_m_xact;
  pLastAnchor->syntaxinfo[2] = syntax_m_data;
  pLastAnchor->syntaxinfo[3] = syntax_m_last;
  pLastAnchor->syntaxinfo[4] = syntax_m_attr;
  pLastAnchor->syntaxinfo[5] = syntax_t_attr;
  pLastAnchor->grouploop.value = grouptyp;

  return (B_E_OK);
}


/* ------------------------------------------------------------------------
 *
 * This function initializes the dynamic capabilities
 * the pointer within the handle struct with the capabilities info
 * is filled and can get accessed afterwards.
 *
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestCapaInit(
    b_handletype handle)
{
  b_int32 lic;                  /* license code */
  b_errtype err;
  size_t num_elem;             /* number of elements in each param block */
  b_param_infotype *p_anchor;   /* just a dummy variable to hold the param
                                 * info */
  b_int32 zwval, zwval1, zwvaldef, zwrule;
  int zwcnt;
  const b_param_stringlisttype * stringtable;
  b_int32 tracememdepth;        /* short cut ... used often */

  b_int32 GUIUsageCode;         /* Ichiban 2.0 limitation concerning 
                                   m_lock and t_lock
                                 */
  
  
  /* SCR; ptrs to elements of handle_array[handle].capable */
  b_dynamic_capability_type *pCapable = handle_array[handle].capable = NULL;
  b_generic_infotype *pMasterblockGen = NULL;
  b_generic_infotype *pMasterAttrGen = NULL;
  b_generic_infotype *pMasterGen = NULL;
  b_generic_infotype *pExerciserGen = NULL;
  b_generic_infotype *pTargetAttrGen = NULL;
  b_generic_infotype *pTargetGenGen = NULL;
  b_generic_infotype *pDecoderGen = NULL;
  b_generic_infotype *pDecoderStdGen = NULL;
  b_generic_infotype *pDecoderExpGen = NULL;
  b_generic_infotype *pDecoderCfg0Gen = NULL;
  b_generic_infotype *pDecoderCfgGen = NULL;
  b_generic_infotype *pDecoderSubGen = NULL;
  b_generic_infotype *pDecoderUcfgGen = NULL;
  b_generic_infotype *pDatamemoryGen = NULL;
  b_generic_infotype *pTraceGenGen = NULL;
  b_generic_infotype *pTracememGen = NULL;
  b_generic_infotype *pPatternGen = NULL;
  b_generic_infotype *pSeqTriggerGenGen = NULL;
  b_generic_infotype *pSeqTriggerParGen = NULL;
  b_generic_infotype *pSeqPerforGenGen = NULL;
  b_generic_infotype *pSeqPerforParGen = NULL;
  b_generic_infotype *pObserverRulesGen = NULL;
  b_generic_infotype *pObserverStatusGen = NULL;
  b_generic_infotype *pMAttrGroupInfoGen = NULL;
  b_generic_infotype *pTAttrGroupInfoGen = NULL;
  b_generic_infotype *pCPUportGen = NULL;
  b_generic_infotype *pStaticIOGen = NULL;
  b_generic_infotype *pPuGen = NULL;
  b_generic_infotype *pBoardGen = NULL;
  b_generic_infotype *pSstCfgGen = NULL;
  b_generic_infotype *pSstGen = NULL;
  b_generic_infotype *pCPCIPinGen = NULL;
  b_generic_infotype *pCPCIStatusGen = NULL;

  void *pVoid;                  /* resolves different addressing systems */

  /* actual hardware ... full description */
  b_hwtype       theBoard;            

  /* but these 2 are the most important */
  b_bool fHasIchiban   = BestHasIchiban(handle);
  b_bool fHasIchiban20 = BestHasIchiban20(handle);
  b_bool fIsDeep = BestIsDeep(handle);
  b_bool fIsCompact = BestIsCompact(handle);
  
  /* Special case: 25b-series has ichiban but no 64 bit ! */
  b_bool fIs25bSeries = BestHwSeriesGet(handle)==B_SERIES_E2925B;

  if (NULL != pCapable)         /* must be zero!! */
    B_ERRETURN(B_E_DYNAMIC_CAPABILITY);

  /* we can't do anything with unknown hardware */
  if(!BestIsKnown(handle))
  {
    BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32) handle_array[handle].port);
    B_ERRETURN(B_E_UNKNOWN_HARDWARE);
  }

  /* get the exact model of board */
  B_ERRCHECK(BestGetHWFromHandle(handle, &theBoard));

  /* the struct itself is allocated here */
  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_dynamic_capability_type)));

  /* short cut */
  pCapable = handle_array[handle].capable = (b_dynamic_capability_type *) pVoid;

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, (size_t) (sizeof(b_int32) * 3)));

  pCapable->capa_code = (b_int32 *) pVoid;

  /* determine the license here ... first one only */
  B_ERRCHECK(BestIHWCapabilityGet(handle, pCapable->capa_code));

  lic = pCapable->capa_code[0]; /* the real license */


  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* ---------------Construct the dynamic Arrays------------------------ */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* ------------------------------------------------------------------------
   * Parameterslists follow here, first master, then target The information
   * is organized as follows: Name, Short name, minimal value, maximal
   * value,default value,rule to follow, Property enum, optional stringlist,
   * [6] syntax infos, License needed. Syntax info contains the
   * optional/required switch for each of the commands
   * (m_block,m_xact,m_data,m_last,m_attr,t_attr) of the xact compiler.
   * -------------------------------------------------------------------- */

  /* -------------------------------------------------------------------- */
  /* master block information (generic and the parameter itself) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pMasterblockGen = pCapable->masterblock_gen = (b_generic_infotype *) pVoid;

  /* group name */
  pMasterblockGen->group_name = "Master Block";

  /* the number of attributes of the master block page */
  pMasterblockGen->num_elem = 0;/* default */

  /* setup non-zero values */
  if (fHasIchiban)
  {
    pMasterblockGen->depth = 255;
    pMasterblockGen->num_pages = 15;  /* page in [1,..,15] in CAPI */
    pMasterblockGen->blocks_per_page = 17;
    pMasterblockGen->num_blocks = pMasterblockGen->num_pages *
      pMasterblockGen->blocks_per_page;
    pMasterblockGen->int_mem_offset = 65536L;
    pMasterblockGen->fixeddelay = 0;  /* no delay between blocks */
  }
  else
  {
    /* Block memory */
    pMasterblockGen->depth = 240;
    pMasterblockGen->num_pages = 127; /* page in [1,..,127] in CAPI */
    pMasterblockGen->blocks_per_page = 16;
    pMasterblockGen->num_blocks = pMasterblockGen->num_pages *
      pMasterblockGen->blocks_per_page;
    pMasterblockGen->int_mem_offset = 65536L;
    pMasterblockGen->fixeddelay = 50; /* 50us delay between blocks */
  }

  /* the masterblock parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER ||
    lic & B_CAPABILITY_HOSTINT) /* this is the prerequisite */
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "busaddr", "bad", 0x0UL, 0xffffffffL, 0UL, NULL, 0UL,
        (b_int32) B_BLK_BUSADDR,
        NULL, par_req, par_req, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "buscmd", "cmd", 0x0UL, 0xfUL, (b_int32) B_CMD_MEM_READ, NULL, 0UL,
        (b_int32) B_BLK_BUSCMD, MasterCmd_table,
        par_req, par_req, par_no, par_no, par_no, par_no, 0UL));

    if (fHasIchiban)
      zwval = 0xff;             /* e2926a can point to a bytenable memory
                                 * with this */
    else
      zwval = 0xf;              /* only standard byten settings for e2925 */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "byten", "ben", 0x0UL, zwval, 0UL, NULL, 0UL, (b_int32) B_BLK_BYTEN,
        byten_table,
        par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

    /* Internal Memory Size */
    if (fHasIchiban)
      zwval = 0xfffffL;
    else
      zwval = 0x1ffffL;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "intaddr", "iad", 0x0UL, zwval, 0UL, NULL,
        (b_int32) B_PARAMRULE_DWORDBOUND, (b_int32) B_BLK_INTADDR, NULL,
        par_req, par_opt, par_no, par_no, par_no, par_no, 0UL));

    /* Burst length */
    if (fHasIchiban)
      zwval = 0x7fffffffL;
    else
      zwval = 0x7fff;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "nofdwords", "nod", 0x1UL, zwval, 1UL, NULL, 0UL,
        (b_int32) B_BLK_NOFDWORDS,
        NULL, par_req, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (lic & B_CAPABILITY_EXERCISER)
      zwval = 0xFF;
    else
      zwval = 0x0;              /* only default attr's allowed */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "attrpage", "apage", 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_BLK_ATTRPAGE,
        NULL, par_opt, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "compflag", "cflag", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_BLK_COMPFLAG, yesno_table,
        par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL))


    if (fHasIchiban)
      zwval = 0xfffffL;         /* e2926a has a bigger internal memory */
    else
      zwval = 0x1ffffL;         /* size of internal memory for the e2925a */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "compoffs", "coffs", 0x0UL, zwval, 0UL, NULL,
        (b_int32) B_PARAMRULE_DWORDBOUND, (b_int32) B_BLK_COMPOFFS,
        NULL, par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

    /* now the new features for the e2926 follow   ........................ */
    /* these new features behave like optional features (see syntax descr) */
    /* some of these features are still commented out because we do not need
     * them in the first step of implementation of the e2925 compatibility */

    if (fHasIchiban)
    {

      if (lic & B_CAPABILITY_EXERCISER)
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "contattr", "contattr", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_BLK_CONTATTR,
            yesno_table,
            par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));


        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "condstart", "cstrt", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_BLK_CONDSTART, yesno_table,
            par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "byten_var", "benv", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_BLK_BYTEN_VAR, yesno_table,
            par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "busaddr_hi", "badhi", 0x0UL, 0xffffffffL, 0UL, NULL, 0UL,
            (b_int32) B_BLK_BUSADDR_HI, NULL,
            par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "dac",
            NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_BLK_BUSDAC, yesno_table,
            par_opt, par_opt, par_no, par_no, par_no, par_no, 0UL));

      }
      /* condstart is not to be used in first step (e2925 compatibility) */
      /* B_ERRCHECK (BestAddParam(&p_anchor,num_elem++, "endblock", "eblk",
       * 0x0UL, 0x1UL, 0x1UL, NULL, 0UL, B_BLK_ENDBLOCK, yesno_table,
       * par_opt,par_opt,par_no,par_no,par_no,par_no, 0UL);
       * 
 */
    }

    /* associate the right number of elements */
    pMasterblockGen->num_elem = num_elem;
  }

  pCapable->masterblock_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the masterattr information (generic and parameter) */
  /* this is the only place where we handle the loop grouping of the */
  /* parameters (except target attributes) */
  /* for every parameter there is a loop bit that is reponsible for it */
  /* to enable ppr to access this information, the structure holds the */
  /* info. we do distinguish between e2925a and e2926a in this list */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pMasterAttrGen = pCapable->masterattr_gen = (b_generic_infotype *) pVoid;

  pMasterAttrGen->group_name = "Master Attribute";  /* group name */

  /* decide depending on hardware what the parameters are ... */
  if(fHasIchiban)
  {
    /* Caution: Because the attributepagesize is variable in Ichiban boards, 
     * the following values describe only the maximum values. 
     * The exact range checking is done in the firmware, not in the CAPI. */
    pMasterAttrGen->depth = 256;          /* depth of the memory */
    pMasterAttrGen->num_pages = 63;       /* !! now variable: 7 or 63 */
    pMasterAttrGen->blocks_per_page = 32; /* !! now variable: 32 or 4 */
    pMasterAttrGen->num_blocks = 252;     /* 224 or 252 */
    pMasterAttrGen->int_mem_offset = 65536L;  /* internal data mem offset
                                               * (xact) */
  }
  else
  {
    pMasterAttrGen->depth = 8160;         /* the depth of the memory */
    pMasterAttrGen->num_pages = 255;      /* page in [1,..,255] in CAPI */
    pMasterAttrGen->blocks_per_page = 32; /* vulgo AttrPageSize */
    pMasterAttrGen->num_blocks = pMasterAttrGen->num_pages *
      pMasterAttrGen->blocks_per_page;
    pMasterAttrGen->int_mem_offset = 65536L;  /* internal data mem offset
                                               * (xact) */
  }

  /* the masterattr parameter information */
  /* set to NULL to make AddParam Fct start with malloc */
  p_anchor = NULL;              
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,  /* not shown in xact
                                                     * compiler */
        "loop", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_DOLOOP, yesno_table,
        par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) B_MATTR_GRP_ML));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "waits", "w",
        0x0UL, 0x1fUL, 0UL, NULL,
        (b_int32) B_PARAMRULE_MAXLONGALLOWED,
        (b_int32) B_M_WAITS, MasterWaits_table, par_no,
        par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_MATTR_GRP_MD0));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "last", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_LAST, yesno_table,
        par_no, par_no, par_no, par_no, par_opt, par_no,
        (b_int32) B_MATTR_GRP_ML));

    if (fHasIchiban)
    {
      zwval = (b_int32) B_MATTR_GRP_MA2;
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "relreq", "rreq",
          (b_int32) B_RELREQ_ON, (b_int32) B_RELREQ_OFF, (b_int32) B_RELREQ_OFF,
          NULL, (b_int32) B_PARAMRULE_NORULE, (b_int32) B_M_RELREQ,
          MasterRelreq_table,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          zwval));
    }
    else
    {
      zwval = (b_int32) B_MATTR_GRP_MA0;
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "relreq", "rreq",
          (b_int32) B_RELREQ_ON, (b_int32) B_RELREQ_OFF, (b_int32) B_RELREQ_OFF,
          NULL, (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_M_RELREQ,
          MasterRelreq_table,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          zwval));
    }


    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "dperr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_DPERR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_MATTR_GRP_MD1));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "dserr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_DSERR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_MATTR_GRP_MD1));


    if (fHasIchiban)
      zwval = (b_int32) B_MATTR_GRP_MA4;
    else
      zwval = (b_int32) B_MATTR_GRP_MA0;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "aperr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_APERR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        zwval));

    if (fHasIchiban)
      zwval = (b_int32) B_MATTR_GRP_MD2;
    else
      zwval = (b_int32) B_MATTR_GRP_MD1;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "dwrpar", "dwp", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_DWRPAR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        zwval));

    if (fHasIchiban)
      zwval = (b_int32) B_MATTR_GRP_MA4;
    else
      zwval = (b_int32) B_MATTR_GRP_MA0;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "awrpar", "awp", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_M_AWRPAR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_no,
        zwval));

    if (fHasIchiban)
      zwval1 = 0x1;             /* fixed back to 1 due to compatibility (was
                                 * 0 before) */
    else
      zwval1 = 0x1;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "waitmode", NULL, 0x0UL, zwval1, 0UL, NULL, 0UL,
        (b_int32) B_M_WAITMODE, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_no,
        (b_int32) B_MATTR_GRP_MD0));

    if (fHasIchiban)
    {
      /* due to compatibility witht the E2925A we
       * still support the 0x1 even if we do not do
       * anything with it */
      zwval = (b_int32) B_MATTR_GRP_MA1;
      zwval1 = 0x1;             
    }
    else
    {
      zwval = (b_int32) B_MATTR_GRP_MA0;
      zwval1 = 0x1;
    }

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "stepmode", NULL, 0x0UL, zwval1, 0UL, NULL, 0UL,
        (b_int32) B_M_STEPMODE, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_no,
        zwval));

    if (fHasIchiban)
      zwval = (b_int32) B_MATTR_GRP_MA2;
    else
      zwval = (b_int32) B_MATTR_GRP_MA0;

    if (!fHasIchiban20)
    {
      /* Ichiban 1.0/1.1 values */
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "lock", NULL, 0x0UL, 0x3UL, 2UL, NULL, 0UL,
          (b_int32) B_M_LOCK, MasterLock_table,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          zwval));
    }
    else
    {
      /* Ichiban 2.0 limitation */
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "lock", NULL, 0x0UL, 0x0UL, 0x0UL, NULL, 0UL,
          (b_int32) B_M_LOCK, MasterLock_table,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          zwval));
    }

    /* now new features for the e2926 follow .............................. */
    /* the corresponding loop bits can now get entered directly */
    /* .................................................................... */
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "steps", NULL, 0x0UL,
          0xffUL, 0UL, NULL, 0UL, (b_int32) B_M_STEPS, NULL, par_no, par_def,
          par_opt, par_opt, par_opt, par_no,
          (b_int32) B_MATTR_GRP_MA1));


      if (!fHasIchiban20)
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "tryback", "back",
        	0x0UL, 0x1UL, 0UL, NULL, 0UL, (b_int32) B_M_TRYBACK, yesno_table,
        	par_no, par_def, par_opt, par_opt, par_opt, par_no,
        	(b_int32) B_MATTR_GRP_MA1));

    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "delay",
        	NULL, 0x0UL, 0x1fffffL, 0UL, NULL, 0UL,
        	(b_int32) B_M_DELAY, NULL,
        	par_no, par_def, par_opt, par_opt, par_opt, par_no,
        	(b_int32) B_MATTR_GRP_MA0));

    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "drelreq", "drreq",
        	(b_int32) B_DRELREQ_ON, (b_int32) B_DRELREQ_OFF, (b_int32) B_DRELREQ_OFF,
        	NULL, (b_int32) B_PARAMRULE_NORULE, (b_int32) B_M_DRELREQ,
        	MasterDRelreq_table, par_no, par_def,
        	par_opt, par_opt, par_opt, par_no,
        	(b_int32) B_MATTR_GRP_MD1));

      }
      else
	  {
    	/* Ichiban 2.0 limitations */
		B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "delay",
        	NULL, 0x2UL, 0x1fffffL, 2UL, NULL, 0UL,
        	(b_int32) B_M_DELAY, NULL,
        	par_no, par_def, par_opt, par_opt, par_opt, par_no,
        	(b_int32) B_MATTR_GRP_MA0));
	  }


      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "dacwp", NULL,
          0x0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_M_DACWRPAR, yesno_table,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          (b_int32) B_MATTR_GRP_MA4));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "dacperr", NULL, 0x0UL,
          0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_M_DACPERR, yesno_table, par_no,
          par_def, par_opt, par_opt, par_opt, par_opt,
          (b_int32) B_MATTR_GRP_MA4));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "resumedelay", "resdel",
          0x2UL, 0x7fUL, 0xaUL, NULL, 0UL,
          (b_int32) B_M_RESUMEDELAY, NULL,
          par_no, par_def, par_opt, par_opt, par_opt, par_no,
          (b_int32) B_MATTR_GRP_MA3));


      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "marker", "mark", 0x0UL,
          0xfUL, 0UL, NULL, 0UL,
          (b_int32) B_M_MARKER, NULL, par_no,
          par_def, par_opt, par_opt, par_opt, par_opt,
          (b_int32) B_MATTR_GRP_MD2));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "repeat", NULL, 0x0UL,
          0xffffffffL, 1UL, NULL, 0UL,
          (b_int32) B_M_REPEAT, NULL, par_no,
          par_def, par_opt, par_opt, par_opt, par_opt,
          (b_int32) B_MATTR_GRP_ML));

      if (lic & B_CAPABILITY_64_BIT)  /* you need 64 Bit license for this */
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "req64", NULL, 0x0UL, 0x1UL,
            0UL, NULL, 0UL,
            (b_int32) B_M_REQ64, yesno_table, par_no,
            par_def, par_opt, par_opt, par_opt, par_no,
            (b_int32) B_MATTR_GRP_MA2));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "awp64", NULL, 0x0UL,
            0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_M_AWRPAR64, yesno_table, par_no,
            par_def, par_opt, par_opt, par_opt, par_no,
            (b_int32) B_MATTR_GRP_MA4));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "dacwp64", NULL, 0x0UL,
            0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_M_DACWRPAR64, yesno_table,
            par_no, par_def, par_opt, par_opt, par_opt, par_no,
            (b_int32) B_MATTR_GRP_MA4));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++, "dwp64", NULL, 0x0UL,
            0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_M_DWRPAR64, yesno_table, par_no,
            par_def, par_opt, par_opt, par_opt, par_no,
            (b_int32) B_MATTR_GRP_MD2));
      }

    }

  }
  /* associate the right number of elements */
  pMasterAttrGen->num_elem = num_elem;
  pCapable->masterattr_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the mastergenericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pMasterGen = pCapable->mastergen_gen = (b_generic_infotype *) pVoid;

  /* leave depth of the master gen memory at 0 (there is no such memory) */
  pMasterGen->group_name = "Master Generic";

  /* the masterattr parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER ||
    lic & B_CAPABILITY_HOSTINT)
  {
    if (!(lic & B_CAPABILITY_EXERCISER))
    {
      zwval = zwval1 = zwvaldef = B_RUNMODE_IMMEDIATE;
    }
    else
    {
      zwval = zwvaldef = B_RUNMODE_IMMEDIATE;
      zwval1 = B_RUNMODE_WONDELAY;
    }
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "runmode", "rmode",
        zwval, zwval1, zwvaldef, NULL, 0UL,
        (b_int32) B_MGEN_RUNMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (!(lic & B_CAPABILITY_EXERCISER))
    {
      zwval = zwval1 = zwvaldef = B_REPEATMODE_SINGLE;
    }
    else
    {
      zwval = B_REPEATMODE_INFINITE;
      zwval1 = zwvaldef = B_REPEATMODE_SINGLE;
    }
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "repeatmode", "repmode",
        zwval, zwval1, zwvaldef, NULL, 0UL,
        (b_int32) B_MGEN_REPEATMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "masterenable", "men", 0x0UL, 0x1UL, 1UL, NULL, 0UL,
        (b_int32) B_MGEN_MASTERENABLE, NULL,
        par_no, par_no, par_no, par_no, par_no, par_no, 0UL));


    if (!(lic & B_CAPABILITY_EXERCISER))
    {
      zwval = zwval1 = zwvaldef = B_ATTRMODE_BLOCK;
    }
    else
    {
      zwval = zwvaldef = B_ATTRMODE_BLOCK;
      zwval1 = B_ATTRMODE_PAGE;
    }
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "attrmode", NULL,
        zwval, zwval1, zwvaldef, NULL, 0UL,
        (b_int32) B_MGEN_ATTRMODE,
        NULL, par_req, par_req, par_no, par_no, par_no, par_no, 0UL));
  }

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
      zwval = 0xffffffffL;
    else
      zwval = 0x0000ffff;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "delayctr", NULL, 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_MGEN_DELAYCTR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "latmode", NULL,
        0x0UL, 0x1UL, 0x01UL, NULL, 0UL, (b_int32) B_MGEN_LATMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "latctr", NULL,
        0x0UL, 0xffUL, 0x0UL, NULL, 0UL, (b_int32) B_MGEN_LATCTR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "mwienable", "mwien", 0x0UL, 0x1UL, 0x1UL, NULL, 0UL,
        (b_int32) B_MGEN_MWIENABLE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));


    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "trybackenable", NULL, 0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_MGEN_TRYBACKENABLE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "disconreload", NULL, 0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_MGEN_DISCONRELOAD,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "retryreload", NULL, 0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_MGEN_RETRYRELOAD,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "invdataenable", NULL, 0UL, 0x1UL, 0x1UL, NULL, 0UL,
          (b_int32) B_MGEN_INVDATAENABLE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "cachelinesize", NULL, 0UL, 0xffUL, 0UL, NULL, 0UL,
          (b_int32) B_MGEN_CACHELINESIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "warn64bit", NULL, 0UL, 1UL, 1UL, NULL, 0UL,
          (b_int32) B_MGEN_WARN64BIT,
          yesno_table, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "restrict64", NULL, 0UL, 1UL, 0UL, NULL, 0UL,
          (b_int32) B_MGEN_64BIT_RESTRICTION_ENABLE,
          yesno_table, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    }
  }
  /* associate the right number of elements */
  pMasterGen->num_elem = num_elem;

  pCapable->mastergen_param = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the exercisergenericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pExerciserGen = pCapable->exercisergen_gen = (b_generic_infotype *) pVoid;

  /* leave depth of the master gen memory at 0 (there is no such memory) */
  pExerciserGen->group_name = "Exerciser Generic";

  /* the exerciser generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "attrpagesize", NULL, 4UL, 32UL, 32UL, NULL,
          (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_EGEN_ATTRPAGESIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    }
  }

  /* associate the right number of elements */
  pExerciserGen->num_elem = num_elem;

  pCapable->exercisergen_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the targetattrproptery information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pTargetAttrGen = pCapable->targetattr_gen = (b_generic_infotype *) pVoid;

  /* group name */
  pTargetAttrGen->group_name = "Target Attribute";

  /* Caution: Because the attributepagesize is variable in an Ichiban board, 
   * the following values describe only the maximum values. 
   * The exact range checking is done in the firmware, not in the CAPI. 
   */
  if(fHasIchiban)
  {
    pTargetAttrGen->depth = 256;
    pTargetAttrGen->num_pages = 63; /* !! now variable: 7 or 63 */
    pTargetAttrGen->blocks_per_page = 32; /* !! now variable: 32 or 4 */
    pTargetAttrGen->num_blocks = 252; /* 224 or 252 */
  }
  else
  {
    pTargetAttrGen->depth = 8160;
    pTargetAttrGen->num_pages = 255;  /* page in [1,..,255] in CAPI */
    pTargetAttrGen->blocks_per_page = 32;
    pTargetAttrGen->num_blocks = pTargetAttrGen->num_pages * pTargetAttrGen->blocks_per_page;
  }

  /* the targetattr parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;
  if (lic & B_CAPABILITY_EXERCISER)
  {
    /* the features for the e2925 come first again ...................... */
    /* this table is identical to the old compiler table in the xact comp */
    /* in some cases there are atttributes in the target attributes that also
     * show up in the master attributes. In this case  the syntax description
     * with the paramter allowances is identical to the above list (master
     * attr). Only writing / range and propertyname differ */
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,  /* not in the xact
                                                     * compiler */
        "loop", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL, (b_int32) B_T_DOLOOP,
        yesno_table, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) B_TATTR_GRP_TC));

    if (fHasIchiban)
      zwval = (b_int32) B_TATTR_GRP_TD1;
    else
      zwval = (b_int32) B_TATTR_GRP_TD0;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "term", NULL, 0x0UL, 0x4UL, 0UL, NULL, 0UL,
        (b_int32) B_T_TERM,
        TargetTerm_table, par_no, par_no, par_no, par_no, par_no, par_opt,
        zwval));

    if(fHasIchiban)
    {
      stringtable = TargetWaits_table;
    }
    else
    {
      stringtable = 0;
    }

    zwval = 0x0UL;       /* everyone else can do 0 waits, as well */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "waits", "w", zwval, 0x1fUL, zwval, NULL,
        (b_int32) B_PARAMRULE_MAXLONGALLOWED, (b_int32) B_T_WAITS,
        stringtable,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_TATTR_GRP_TD0));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "dperr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_T_DPERR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_TATTR_GRP_TD1));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "dserr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_T_DSERR, yesno_table,
        par_no, par_no, par_no, par_no, par_no, par_opt,
        (b_int32) B_TATTR_GRP_TD1));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "aperr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_T_APERR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_TATTR_GRP_TA0));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "wrpar", "wp", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_T_WRPAR, yesno_table,
        par_no, par_def, par_opt, par_opt, par_opt, par_opt,
        (b_int32) B_TATTR_GRP_TD1));

    /* now the new features for the e2926 follow   ........................ */
    /* these new features behave like optional features (see syntax descr) */
    /* some of these features are still dead (LICENSE_NO) because we do not
     * need them in the first step of implementation of the e2925
     * compatibility */

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "dacperr", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_T_DACPERR, yesno_table,
          par_no, par_no, par_no, par_no, par_no, par_opt,
          (b_int32) B_TATTR_GRP_TA0));

      /* B_ERRCHECK (BestAddParam(&p_anchor,num_elem++, "waitmode", NULL,
       * 0x0UL, 0x1UL, 0UL, NULL, 0UL, B_T_WAITMODE, yesno_table,
       * par_no,par_no,par_no,par_no,par_no,par_opt, 0)); ** not supported
       * (does not work, hw problem ) */

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "marker", "mark", 0x0UL, 0xfUL, 0UL, NULL, 0UL,
          (b_int32) B_T_MARKER, NULL,
          par_no, par_def, par_opt, par_opt, par_opt, par_opt,
          (b_int32) B_TATTR_GRP_TD2));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "repeat", NULL, 0x0UL, 0xffffffffL, 0x1UL, NULL, 0UL,
          (b_int32) B_T_REPEAT, NULL,
          par_no, par_def, par_opt, par_opt, par_opt, par_opt,
          (b_int32) B_TATTR_GRP_TC));

      if (lic & B_CAPABILITY_64_BIT)  /* you need 64 Bit license for this */
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "ack64", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_T_ACK64, yesno_table,
            par_no, par_no, par_no, par_no, par_no, par_opt,
            (b_int32) B_TATTR_GRP_TA0));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "wp64", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
            (b_int32) B_T_WRPAR64, yesno_table,
            par_no, par_no, par_no, par_no, par_no, par_opt,
            (b_int32) B_TATTR_GRP_TD1));
      }
    }

  }
  /* associate the right number of elements */
  pTargetAttrGen->num_elem = num_elem;

  pCapable->targetattr_param = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the targetgenericproperty information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pTargetGenGen = pCapable->targetgen_gen = (b_generic_infotype *) pVoid;

  /* leave depth of the target gen memory = 0 (there is no such memory) */
  pTargetGenGen->group_name = "Target Generic";

  /* the targetgen parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER ||
    lic & B_CAPABILITY_HOSTINT)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "runmode", "rmode", 0x0UL, 0x1UL,
        (b_int32) B_RUNMODE_SEQUENTIAL,
        NULL, 0UL, (b_int32) B_TGEN_RUNMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "memspace", "mspace", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TGEN_MEMSPACE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "iospace", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL, (b_int32) B_TGEN_IOSPACE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "romenable", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TGEN_ROMENABLE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    /* the new features for the e2926         ........................ */
    if (fHasIchiban)
    {

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "backcapable", NULL, 0x0UL, 0x1UL, 0UL, NULL, 0UL,
          (b_int32) B_TGEN_BACKCAPABLE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      /* B_ERRCHECK (BestAddParam(&p_anchor,num_elem++, "prefetch_mode",
       * "prefetch", 0x0UL, 0x1UL, 0UL, NULL, 0UL, (b_int32)B_TGEN_PREFETCH,
       * yesno_table, par_no,par_no,par_no,par_no,par_no,par_no, 0UL)); */
    }

  }

  /* associate the right number of elements */
  pTargetGenGen->num_elem = num_elem;

  pCapable->targetgen_param = p_anchor; /* set it */



  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------Start of Decoder--------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */

  /* --------------------------------------------------------------------
   * Decoder information is stored in two hierarchical layers: the generic
   * info for all decoders, which gives information about available
   * decodertypes, and the corresponding dyncap parameters. The second layer
   * contains more specific info about the decodertypes */
  /* -------------------------------------------------------------------- */
  /* This information (generic and parameters)				 */
  /* decoders are organized in 6 standard decoder, one expansion rom      *
   * decoder, one config type 0 decoder, one subtractive decoder, one     *
   * config decoder and a user config decoder. There is NO range checking *
   * done in the C-API due to the function call used. The dependencies    *
   * within the parameters of a decoder itself are not reflected in the   *
   * ranges shown here.                                                   *
   * -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderGen = pCapable->decoder_gen = (b_generic_infotype *) pVoid;

  pDecoderGen->group_name = "Decoder";

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER ||
    lic & B_CAPABILITY_HOSTINT)
  {

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Standard 1", "std1",
        0x0UL, 0x0UL, 0UL, NULL,
        (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) B_PARAM_DECODER_STD));

    if (lic & B_CAPABILITY_EXERCISER)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "Standard 2", "std2",
          0x0UL, 0x0UL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) B_PARAM_DECODER_STD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "Standard 3", "std3",
          0x0UL, 0x0UL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_3,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) B_PARAM_DECODER_STD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "Config Type 0", "config",
          0x0UL, 0x0UL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_CONFIG,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) B_PARAM_DECODER_CFG0));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "Expansion ROM", "exprom",
          0x0UL, 0x0UL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_EXPROM,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) B_PARAM_DECODER_EXP));

      if (fHasIchiban)
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "Standard 4", "std4",
            0x0UL, 0x0UL, 0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_4,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) B_PARAM_DECODER_STD));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "Standard 5", "std5",
            0x0UL, 0x0UL, 0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_5,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) B_PARAM_DECODER_STD));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "Standard 6", "std6",
            0x0UL, 0x0UL, 0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_STANDARD_6,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) B_PARAM_DECODER_STD));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "Config", "fullconfig",
            0x0UL, 0x0UL, 0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_FULL_CONFIG,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) B_PARAM_DECODER_CFG));

/*------------ the user config decoder isn't shown to anyone - ever!
  B_ERRCHECK (BestAddParam(&p_anchor,num_elem++,
  "User Config", "userconfig",
  0x0UL, 0x0UL, 0UL, NULL,
  (b_int32)B_PARAMRULE_NORULE, (b_int32)B_DEC_USER_CONFIG,
  NULL, par_no,par_no,par_no,par_no,par_no,par_no,
  (b_int32)B_PARAM_DECODER_UCFG));

  -----------*/
	
	B_ERRCHECK (BestAddParam(&p_anchor,num_elem++,
	    "Subtractive", "subtract",
	    0x0UL, 0x0UL, 0UL, NULL,
	    (b_int32)B_PARAMRULE_NORULE, (b_int32)B_DEC_SUBTRACTIVE,
	    NULL, par_no,par_no,par_no,par_no,par_no,par_no,
	    (b_int32)B_PARAM_DECODER_SUB));
      }
    }
  }

  /* associate the right number of elements */
  pDecoderGen->num_elem = num_elem;

  pCapable->decoder_param = p_anchor; /* set it */




  /*----------------------------------------------------------------------*
   *------------------- info about standard decoders ---------------------*
   *----------------------------------------------------------------------*/

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderStdGen = pCapable->decoder_std_gen = (b_generic_infotype *) pVoid;

  pDecoderStdGen->group_name = "Standard Decoder";

  if (fHasIchiban)
    pDecoderStdGen->depth = 6;
  else
    pDecoderStdGen->depth = 3;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER ||
    lic & B_CAPABILITY_HOSTINT)
  {
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "size", "size",
          5UL, 64UL, 12UL, NULL,
          (b_int32) B_PARAMRULE_ZEROALLOWED, (b_int32) B_DEC_SIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));
    }
    else
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "size", "size",
          4UL, 24UL, 12UL, NULL,
          (b_int32) B_PARAMRULE_ZEROALLOWED, (b_int32) B_DEC_SIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));
    }


    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "baseaddr", "base",
        0x0UL, 0xffffffffL, 0UL, NULL,
        (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEADDR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) DEC_PROP_HARD));

    if (fHasIchiban20)
	{
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "speed", "speed",
        (b_int32) B_DSP_SLOW,
        (1<<B_DSP_SLOW)|(1<<B_DSP_MEDIUM)|(1<<B_DSP_NODEVSEL),
        (b_int32) B_DSP_MEDIUM, NULL,
          (b_int32) B_PARAMRULE_BITSET, (b_int32) B_DEC_SPEED,
          DecoderSpeed_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));
   	}
	else
	{
      if (fHasIchiban)
    	zwval = (b_int32) B_DSP_NODEVSEL;
      else
    	zwval = (b_int32) B_DSP_MEDIUM;

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "speed", "speed",
          (b_int32) (b_int32) B_DSP_SLOW, zwval, (b_int32) B_DSP_MEDIUM, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
          DecoderSpeed_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));
    }

    /* the mode property is obsolete and therefore not supported by dynamic
     * capabilities */

    if (fHasIchiban)
      zwval = (b_int32) B_PARAMRULE_NORULE;
    else
      zwval = (b_int32) B_PARAMRULE_EVEN;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "location", "loc",
        0x0UL, 0x04UL, 0UL, NULL,
        zwval, (b_int32) B_DEC_LOCATION,
        DecoderLocation_table,
        par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) DEC_PROP_HARD));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "prefetch", "prefetch",
        0x0UL, 0x1UL, 0UL, NULL,
        (b_int32) (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_PREFETCH,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) DEC_PROP_HARD));

    /* the new features for the e2926         ........................ */
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "behavior", "behavior",
          (b_int32) B_BEH_NORMAL, (b_int32) B_BEH_CUSTOM, (b_int32) B_BEH_NORMAL,
          NULL,
          (b_int32) (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BEHAVIOR,
          DecoderBehavior_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_VIEW));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "basedec", "basedec",
          (b_int32) B_DEC_STANDARD_1, (b_int32) B_DEC_STANDARD_5,
          (b_int32) B_DEC_STANDARD_1, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEDEC,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_DATA, (b_int32) B_RES_INTBUS, (b_int32) B_RES_DATA,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 12UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "command", "cmd",
          0x0UL, 0xffffUL, 0xd0c0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BUSCMD,
          DecoderCommands_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "dac", "dac",
          (b_int32) B_DAC_NO, (b_int32) B_DAC_YES, (b_int32) B_DAC_NO, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_DAC,
          DecoderDAC_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "idsel", "idsel",
          (b_int32) B_IDSEL_ASSERT, (b_int32) B_IDSEL_DONTCARE, (b_int32) B_IDSEL_DONTCARE,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_IDSEL,
          DecoderIdsel_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "mask", "mask",
          0x0UL, 0xffffffffL, 0xfffff000L, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_MASK,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      if (lic & B_CAPABILITY_64_BIT)  /* you need 64 Bit license for this */
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "high_baseaddr", "basehi",
            0x0UL, 0xffffffffL, 0x0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEADDR_HI,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) DEC_PROP_HARD));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "high_mask", "maskhi",
            0x0UL, 0xffffffffL, 0x0UL, NULL,
            (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_MASK_HI,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) DEC_PROP_OTHER));
      }
    }
  }

  /* associate the right number of elements */
  pDecoderStdGen->num_elem = num_elem;

  pCapable->decoder_std_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for expansion rom decoder */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderExpGen = pCapable->decoder_exp_gen = (b_generic_infotype *) pVoid;

  /* the number of expansion rom decoders in e2925 and e2926 */
  pDecoderExpGen->depth = 1;    /* only one */
  pDecoderExpGen->group_name = "Expansion Rom Decoder";

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
      zwrule = (b_int32) B_PARAMRULE_NORULE;
    else
      zwrule = (b_int32) B_PARAMRULE_MINMAXONLY;

    if (fHasIchiban)
      zwval = 32;
    else
      zwval = 18;

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "size", "size",
        0x0UL, zwval, 16UL, NULL,
        zwrule, (b_int32) B_DEC_SIZE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) DEC_PROP_HARD));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "baseaddr", "base",
        0x0UL, 0xffffffffL, 0UL, NULL,
        (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEADDR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) DEC_PROP_HARD));

    if (fHasIchiban20)
	{
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "speed", "speed",
        (b_int32) B_DSP_SLOW,
        (1<<B_DSP_SLOW)|(1<<B_DSP_MEDIUM)|(1<<B_DSP_NODEVSEL),
        (b_int32) B_DSP_MEDIUM, NULL,
          (b_int32) B_PARAMRULE_BITSET, (b_int32) B_DEC_SPEED,
          DecoderSpeed_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));
   	}
	else
	{
      if (fHasIchiban)
      {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
        	(b_int32) B_DSP_SLOW, (b_int32) B_DSP_NODEVSEL, (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	(b_int32) DEC_PROP_SOFT));
      }
      else
      {                           /* make invisible to target decode window! */
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
        	(b_int32) B_DSP_SLOW, (b_int32) B_DSP_MEDIUM, (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	(b_int32) DEC_PROP_OTHER));
      }
    }
    /* the new features for the e2926         ........................ */
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_DATA, (b_int32) B_RES_INTBUS, (b_int32) B_RES_EXPROM,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 16UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "mask", "mask",
          0x0UL, 0xffffffffL, 0xffff0000L, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_MASK,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));
    }
  }
  /* associate the right number of elements */
  pDecoderExpGen->num_elem = num_elem;

  pCapable->decoder_exp_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for configuration type 0 decoder */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderCfg0Gen = pCapable->decoder_cfg0_gen = (b_generic_infotype *) pVoid;

  /* the number of config type 0 decoders in e2925 and e2926 */
  pDecoderCfg0Gen->depth = 1;   /* only one */
  pDecoderCfg0Gen->group_name = "Config Type 0 Decoder";

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if ((handle_array[handle].port == B_PORT_PCI_CONF) ||
    (handle_array[handle].port == B_PORT_PCI_IO))
  {
    zwval = (b_int32) DEC_PROP_VIEW;
  }

  else
    zwval = (b_int32) DEC_PROP_OTHER;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "size", "size",
        0x0UL, 0x08UL, 0x08UL, NULL,
        (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_DEC_SIZE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_HARD));

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "baseaddr", "base",
          0x0UL, 0xffffffffL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEADDR,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_HARD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "mask", "mask",
          0x0UL, 0xffffffffL, 0x03UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_MASK,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_OTHER));

      if (fHasIchiban20)
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
          (b_int32) B_DSP_SLOW,
          (1<<B_DSP_SLOW)|(1<<B_DSP_MEDIUM)|(1<<B_DSP_NODEVSEL),
          (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_BITSET, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_SOFT));
   	  }
	  else
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
        	(b_int32) B_DSP_SLOW, (b_int32) B_DSP_NODEVSEL, (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_SOFT));
      }

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_DATA, (b_int32) B_RES_INTBUS, (b_int32) B_RES_REGFILE,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 8UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          zwval == (b_int32) DEC_PROP_VIEW ? (b_int32) DEC_PROP_VIEW : (b_int32) DEC_PROP_SOFT));
    }
  }
  /* associate the right number of elements */
  pDecoderCfg0Gen->num_elem = num_elem;

  pCapable->decoder_cfg0_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for config decoder */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderCfgGen = pCapable->decoder_cfg_gen = (b_generic_infotype *) pVoid;

  pDecoderCfgGen->group_name = "Config Decoder";

  if (fHasIchiban)              /* 2925 has no decoders */
    pDecoderCfgGen->depth = 1;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "size", "size",
          0x0UL, 0x08UL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_DEC_SIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "baseaddr", "base",
          0x0UL, 0xffffffffL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BASEADDR,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "mask", "mask",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_MASK,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      if (fHasIchiban20)
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
          (b_int32) B_DSP_SLOW,
          (1<<B_DSP_SLOW)|(1<<B_DSP_MEDIUM)|(1<<B_DSP_NODEVSEL),
          (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_BITSET, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) DEC_PROP_SOFT));
   	  }
	  else
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
        	(b_int32) B_DSP_SLOW, (b_int32) B_DSP_NODEVSEL, (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	(b_int32) DEC_PROP_SOFT));
      }
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_DATA, (b_int32) B_RES_INTBUS, (b_int32) B_RES_REGFILE,
          NULL,
          (b_int32) B_PARAMRULE_EVEN, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 8UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "primary_bus", "primary",
          0x0UL, 255UL, 0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_PRIMARY_BUS,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "secondary_bus", "secondary",
          0x0UL, 255UL, 255UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SECONDARY_BUS,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));
    }
  }
  /* associate the right number of elements */
  pDecoderCfgGen->num_elem = num_elem;

  pCapable->decoder_cfg_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for subtractive decoder */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderSubGen = pCapable->decoder_sub_gen = (b_generic_infotype *) pVoid;

  pDecoderSubGen->group_name = "Subtractive Decoder";

  if (fHasIchiban)              /* 2925 has no decoder */
    pDecoderSubGen->depth = 1;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "size", "size",
          0x0UL, 0x1UL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_DEC_SIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));

      if (fHasIchiban20)
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
          (b_int32) B_DSP_SLOW,
          (1<<B_DSP_SLOW)|(1<<B_DSP_MEDIUM)|(1<<B_DSP_NODEVSEL),
          (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_BITSET, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
            (b_int32) DEC_PROP_SOFT));
   	  }
	  else
	  {
    	B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        	"speed", "speed",
        	(b_int32) B_DSP_SLOW, B_DSP_NODEVSEL, (b_int32) B_DSP_MEDIUM, NULL,
        	(b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_SPEED,
        	DecoderSpeed_table,
        	par_no, par_no, par_no, par_no, par_no, par_no,
        	(b_int32) DEC_PROP_SOFT));
      }


      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "command", "cmd",
          0x0UL, 0xfUL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_BUSCMD,
          DecoderCommands_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "dac", "dac",
          (b_int32) B_DAC_NO, (b_int32) B_DAC_BOTH, (b_int32) B_DAC_BOTH, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_DAC,
          DecoderDAC_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "idsel", "idsel",
          (b_int32) B_IDSEL_ASSERT, (b_int32) B_IDSEL_DONTCARE, (b_int32) B_IDSEL_DONTCARE,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_IDSEL,
          DecoderIdsel_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_OTHER));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_DATA, (b_int32) B_RES_INTBUS, (b_int32) B_RES_DATA,
          NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 2UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));
    }
  }
  /* associate the right number of elements */
  pDecoderSubGen->num_elem = num_elem;

  pCapable->decoder_sub_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the decoder information for the user config decoder */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDecoderUcfgGen = pCapable->decoder_ucfg_gen = (b_generic_infotype *) pVoid;

  pDecoderUcfgGen->group_name = "User Config Decoder";

  if (fHasIchiban)
    pDecoderUcfgGen->depth = 1;

  /* the decoder parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_EXERCISER)
  {
    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "size", "size",
          0x0UL, 0x1UL, 0x1UL, NULL,
          (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_DEC_SIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_HARD));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resource", "res",
          (b_int32) B_RES_REGFILE, (b_int32) B_RES_CPUHELP, (b_int32) B_RES_REGFILE, NULL,
          (b_int32) B_PARAMRULE_MINMAXONLY, (b_int32) B_DEC_RESOURCE,
          DecoderResource_table,
          par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "resbase", "resbase",
          0x0UL, 0xffffffffL, 0x0UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESBASE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ressize", "ressize",
          2UL, 23UL, 2UL, NULL,
          (b_int32) B_PARAMRULE_NORULE, (b_int32) B_DEC_RESSIZE,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) DEC_PROP_SOFT));
    }
  }
  /* associate the right number of elements */
  pDecoderUcfgGen->num_elem = num_elem;

  pCapable->decoder_ucfg_param = p_anchor;  /* set it */


  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------End of Decoder----------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */
  /*--------------------------------------------------------------------- */

  /* -------------------------------------------------------------------- */
  /* the datamemory information (generic only) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pDatamemoryGen = pCapable->datamemory_gen = (b_generic_infotype *) pVoid;

  pDatamemoryGen->group_name = "Data Memory";

  /* int_mem_offset is always 65536L */
  pDatamemoryGen->int_mem_offset = 65536L;

  if (fHasIchiban)
  {
    pDatamemoryGen->depth = 0x80000L; /* 512 KB */
    pDatamemoryGen->num_blocks = 0x7; /* qword alignment with busaddress for
                                       * transfers */
  }
  else
  {
    pDatamemoryGen->depth = 0x20000L; /* 128 KB */
    pDatamemoryGen->num_blocks = 0x3; /* dword alignment with busaddress for
                                       * transfers */
  }

  /* -------------------------------------------------------------------- */
  /* the trace generic property information (generic and parameters) */
  /* moved after the tracemem stuff so we know the tracememsize */
  /* -------------------------------------------------------------------- */


  /* -------------------------------------------------------------------- */
  /* the tracememoryprop information (generic and signals) */
  /* the number of bytes per trace memory line is stored in num_pages */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pTracememGen = pCapable->tracemem_gen = (b_generic_infotype *) pVoid;

  pTracememGen->group_name = "Trace Memory Signals";

  /* NOTE; the number of bytes per line (0) is in the num_pages field */

  if (fHasIchiban)
    pTracememGen->num_pages = 20;
  else
    pTracememGen->num_pages = 12;

  /* associate the right number of elements */

  /* every board has another depth of the tracememory */
  /* and this depth is also dependend on license information */

  if(fHasIchiban)
  {
    if (handle_array[handle].port == B_PORT_OFFLINE)
    {
      /* use the license to determine the maximal size of the trace memory */
      if (lic & B_CAPABILITY_TRACEDEPTH_MASK)
      {
	    tracememdepth = 0x8000UL << 
                        (int) ((lic & B_CAPABILITY_TRACEDEPTH_MASK) >> 
                         B_CAPABILITY_TRACEDEPTH_SHIFT);
      }
	  else
      {
        tracememdepth = 0x0UL;
      }
	  
      if(fIsDeep)
      {
        if (tracememdepth > 4194304L)
        {
          tracememdepth = 4194304L;
        }
		pTracememGen->depth = tracememdepth;
      }
      else
      {
        if (tracememdepth > 65536L)
        {
		  tracememdepth = 65536L;
        }
		pTracememGen->depth = tracememdepth;
      }
    }
    else
      {
      /* assume no hardware */
        tracememdepth = 0;
      
#ifndef E2921A_MIN
      /* if we have real hardware, ask the hardware */
      (void)BestTraceMemorySizeGet(handle, &tracememdepth);
#endif  /* E2921A_MIN */
      
      pTracememGen->depth = tracememdepth;
    }

    if (theBoard == B_HW_E2925B  /* special case, supporting E2925B */
	|| theBoard == B_HW_E2925B_DEEP) /* including deep trace version */
    {
      pCapable->tracemem_param = e2925b_signal_table;
    }
    else
    {
	  if (fHasIchiban20)
	  {
        pCapable->tracemem_param = e2928a_signal_table;
      }
	  else
	  {
        pCapable->tracemem_param = e2926a_signal_table;
	  }
    }
  }
  else  /* no ichiban */
  {
    if (fIsDeep)
    {
      pTracememGen->depth = 1048576L;
      tracememdepth = pTracememGen->depth;
      pCapable->tracemem_param = e2925a_deep_signal_table;
    }
    else
    {
      pTracememGen->depth = 32768L;
      tracememdepth = pTracememGen->depth;
      pCapable->tracemem_param = e2925a_signal_table;
    }
  }

  /* count the number of elements within the list of signals */
  /* every list has a terminating line (important !! ) */
  num_elem = 0;
  if (pCapable->tracemem_param)
  {
    while (pCapable->tracemem_param[(int) num_elem++].param_name);  /*lint !e722 */
  }

  pTracememGen->num_elem = num_elem - 1;


  /* -------------------------------------------------------------------- */
  /* MUST BE AFTER TRACEMEM stuff!!!! VERY important to access tracememdepth */
  /* the trace generic property information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pTraceGenGen = pCapable->tracegen_gen = (b_generic_infotype *) pVoid;

  pTraceGenGen->group_name = "Trace Memory";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "HEARTBEATMODE", "hbmode", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TRC_HEARTBEATMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (fHasIchiban)
    {
      zwval = 0xffffffffL;      /* e2926a has a deeper heartbeatcounter */
    }
    else
    {
      zwval = 0xffff;           /* e2925 has just a 16 bit hbctr (and 256
                                 * granularity) */
    }
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "HEARTBEATVALUE", "hbvalue", 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_TRC_HEARTBEATVALUE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (theBoard == B_HW_E2925A)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			      "PERFANALYZER_MODE", "amode",
			      B_E2925_COMPATIBLE, B_E2925_COMPATIBLE,
			      B_E2925_COMPATIBLE, NULL,
			      0UL,
			      (b_int32) B_TRC_PERFANALYZER_MODE,
			      NULL, par_no, par_no, par_no, par_no,
			      par_no, par_no, 0UL));

    }
    else
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			      "TRIG_HISTORY", "trighist", 0x0UL, tracememdepth,
			      tracememdepth / 2, NULL,
			      (b_int32) B_PARAMRULE_DWORDBOUND,
			      (b_int32) B_TRC_TRIG_HISTORY,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "PERFANALYZER_MODE", "amode", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TRC_PERFANALYZER_MODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "PERFPATT0_MODE", "pt0mode", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TRC_PERFPATT0_MODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TRIG15IO_MODE", "trigiomode", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TRC_TRIG15IO_MODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    }
  }

  /* associate the right number of elements */
  pTraceGenGen->num_elem = num_elem;

  pCapable->tracegen_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the patternproperty information (generic and signals) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pPatternGen = pCapable->pattern_gen = (b_generic_infotype *) pVoid;

  pPatternGen->group_name = "Pattern Generic";

  /* every board has another number of patterns */
  if (fHasIchiban)
  {
    pPatternGen->depth = 24;

    /* in this case we use the public table (pattdefs.h) to create a pattern
     * table for dynamic capabilities */


    p_anchor = NULL;            /* set to NULL to make AddParam Fct start
                                 * with malloc */
    num_elem = 0;
    zwcnt = 0;

    if (lic & B_CAPABILITY_ANALYZER)
    {
	  const b_patt_infotype *patt_signal_table=(fIs25bSeries?e2925b_patt_signal_table:e2926a_patt_signal_table);
	   
	  while (patt_signal_table[zwcnt].Name)
      {
        if (patt_signal_table[zwcnt].GUIUsageCode) /* only if used at all */
        {
 
           if (fHasIchiban20 && 
		       (strcmp (patt_signal_table[zwcnt].Name,"m_lock")==0 ||          
		        strcmp (patt_signal_table[zwcnt].Name,"t_lock")==0))
		   {          
             /* Ichiban 2.0 limitation:
			    Do not show m_lock and t_lock in GUI, but leave behavior for FW parser 
				unchanged.
			  */
              GUIUsageCode=0; /* no GUI */
		   }
           else
           {
              GUIUsageCode=patt_signal_table[zwcnt].GUIUsageCode;
           }
 
 
           B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              patt_signal_table[zwcnt].Name,
              NULL,
              patt_signal_table[zwcnt].MinVal,
              patt_signal_table[zwcnt].MaxVal,
              patt_signal_table[zwcnt].BitCount,
              NULL,
              GUIUsageCode,
              0x0UL,
              patt_signal_table[zwcnt].StringList,
              par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
        }
        zwcnt++;
      }	  
    }

    /* associate the right number of elements */
    pPatternGen->num_elem = num_elem;

    pCapable->pattern_param = p_anchor; /* set it */
  }
  else
  {
    if (fIsDeep)
    {
      pPatternGen->depth = 12;
      pCapable->pattern_param = e2925a_deep_patt_signal_table;
    }
    else
    {
      pPatternGen->depth = 2;
      pCapable->pattern_param = e2925a_patt_signal_table;
    }
  }

  /* count the number of elements within the list of signals */
  /* every list has a terminating line (important !! ) */
  if (BestIs2925(handle))
  {
    num_elem = 0;
    if (pCapable->pattern_param)
    {
      while (pCapable->pattern_param[(int) num_elem++].param_name); /*lint !e722 */
    }

    pPatternGen->num_elem = num_elem - 1;
  }

  /* -------------------------------------------------------------------- */
  /* the trigger sequencer generic information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSeqTriggerGenGen = pCapable->seqtriggergen_gen = (b_generic_infotype *) pVoid;

  pSeqTriggerGenGen->group_name = "Trigger Sequencer Generic";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if ((lic & B_CAPABILITY_ANALYZER) && (theBoard != B_HW_E2925A))
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "CTRC_PREL", "cprel", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_TRIGSEQGEN_CTRC_PREL,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pSeqTriggerGenGen->num_elem = num_elem;

  pCapable->seqtriggergen_param = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* the trigger sequencer parameter information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSeqTriggerParGen = pCapable->seqtriggerpar_gen = (b_generic_infotype *) pVoid;

  pSeqTriggerParGen->group_name = "Trigger Sequencer";

  if (theBoard == B_HW_E2925A_DEEP)
    pSeqTriggerParGen->depth = 252;
  else if (fHasIchiban)
    pSeqTriggerParGen->depth = 256;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if ((lic & B_CAPABILITY_ANALYZER) && (theBoard != B_HW_E2925A))
  {
    if (fHasIchiban)
      zwval = 0xf;              /* e2926a has a bigger sequencer memory */
    else
      zwval = 0x4;              /* how many states in the e2925a deep trace
                                 * sequ */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "State", "state", 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_TRIGSEQ_STATE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Next State", "nextstate", 0x0UL, zwval, 0x0UL, NULL, 0UL,
        (b_int32) B_TRIGSEQ_NEXTSTATE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    /* now a switch in prop types ... */
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Transfer Condition", "x", 0x0UL, 0x0UL, 0x0UL, "1", 0x0UL,
        (b_int32) B_TRIGSEQ_XCOND,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Trigger Condition", "trig", 0x0UL, 0x0UL, 0UL, "0", 0UL,
        (b_int32) B_TRIGSEQ_TRIGCOND,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Storage Condition", "sq", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
        (b_int32) B_TRIGSEQ_SQCOND,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Count Enable Condition", "ce", 0x0UL, 0x0UL, 0UL, "0", 0UL,
        (b_int32) B_TRIGSEQ_CE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Counter Load Condition", "cload", 0x0UL, 0x0UL, 0UL, "0", 0UL,
        (b_int32) B_TRIGSEQ_CLOAD,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pSeqTriggerParGen->num_elem = num_elem;

  pCapable->seqtriggerpar_param = p_anchor; /* set it */


  /* -------------------------------------------------------------------- */
  /* the performance sequencer generic information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSeqPerforGenGen = pCapable->seqperforgen_gen = (b_generic_infotype *) pVoid;

  pSeqPerforGenGen->group_name = "Performance Sequencer Generic";

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if ((lic & B_CAPABILITY_ANALYZER) && (theBoard != B_HW_E2925A))
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "CTRC_PREL", "cprel", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_PERFGEN_CTRC_PREL,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "CAMODE", "mode", 0x0UL, 0x1UL, 0UL, NULL, 0UL,
        (b_int32) B_PERFGEN_CAMODE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pSeqPerforGenGen->num_elem = num_elem;

  pCapable->seqperforgen_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the performance sequencer parameter information (generic and parameters) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSeqPerforParGen = pCapable->seqperforpar_gen = (b_generic_infotype *) pVoid;

  pSeqPerforParGen->group_name = "Performance Sequencer";

  if (theBoard == B_HW_E2925A_DEEP)
    pSeqPerforParGen->depth = 64;
  else if (fHasIchiban)
    pSeqPerforParGen->depth = 256;

  /* the triggerseq parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if ((lic & B_CAPABILITY_ANALYZER) && (theBoard != B_HW_E2925A))
  {
    if (fHasIchiban)
      zwval = 0xf;              /* e2926a has a bigger sequencer memory */
    else
      zwval = 0x4;              /* how many states in the e2925a deep trace
                                 * sequ */

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "State", "state", 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_PERFSEQ_STATE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Next State", "nextstate", 0x0UL, zwval, 0UL, NULL, 0UL,
        (b_int32) B_PERFSEQ_NEXTSTATE,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    /* now a switch in prop types ... */
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Transfer Condition", "x", 0x0UL, 0x0UL, 0x0UL, "1", 0x0UL,
        (b_int32) B_PERFSEQ_XCOND,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Counter A Condition", "caen", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
        (b_int32) B_PERFSEQ_CA_EN,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Counter B Condition", "cben", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
        (b_int32) B_PERFSEQ_CB_EN,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Counter C Condition", "ccen", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
        (b_int32) B_PERFSEQ_CC_EN,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "Counter C Load", "ccload", 0x0UL, 0x0UL, 0x0UL, "0", 0x0UL,
        (b_int32) B_PERFSEQ_CC_LOAD,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pSeqPerforParGen->num_elem = num_elem;

  pCapable->seqperforpar_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the observer rules information (generic and the rules itself) */
  /* the paramrule information holds the info for the bit position ! */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pObserverRulesGen = pCapable->observerrules_gen = (b_generic_infotype *) pVoid;

  /* leave depth of the observer rules at 0 (what else would make sense?) */
  pObserverRulesGen->group_name = "Observer Rules";

  /* the observerrules rules information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "FRAME 0", "FRAME 0:Whenever STOP# is asserted, "
        "the master must deassert FRAME# as soon as IRDY# "
        "can be asserted. (PCI Spec. Appendix C, Rule 12 e)",
        0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_R_FRAME_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "FRAME 1", "FRAME 1:Fast Back-To-Back is only allowed after a "
        "WRITE transaction or a Master abort. (PCI Spec. "
        "Sect. 3.4.2. Fast Back-to-Back Transaction)", 0UL, 1UL, 0UL, NULL, 0x1UL,
        (b_int32) B_R_FRAME_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "IRDY 0", "IRDY 0:IRDY# must not be asserted on the same clock "
        "edge that FRAME# is asserted, but one or more clocks "
        "later. (PCI Spec. Sect. 3.3.1. Read Transaction - "
        "see timing diagram)", 0UL, 1UL, 0UL, NULL, 2UL,
        (b_int32) B_R_IRDY_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "IRDY 1", "IRDY 1:FRAME# can not be deasserted unless IRDY# "
        "is asserted. IRDY# must always be asserted on the "
        "first clock edge that FRAME# is deasserted. "
        "(PCI Spec. Appendix C, Rule 8 c)", 0UL, 1UL, 0UL, NULL, 3UL,
        (b_int32) B_R_IRDY_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "IRDY 2", "IRDY 2:IRDY# must be deasserted after last transfer "
        "or when FRAME# is high and STOP was asserted. "
        "(PCI Spec. Sect. 3.3.3.1. and 3.3.3.2.1, Rule 1)", 0UL, 1UL, 0UL, NULL, 4UL,
        (b_int32) B_R_IRDY_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "IRDY 3", "IRDY 3:A transaction starts when FRAME# is sampled "
        "asserted for the first time - IRDY# must not go low "
        "when FRAME# is high. (PCI Spec. Appendix C, Rule 7)", 0UL, 1UL, 0UL, NULL, 5UL,
        (b_int32) B_R_IRDY_3,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "IRDY 4", "IRDY 4:Once a master has asserted IRDY# it cannot "
        "change IRDY# or FRAME# until the current data phase "
        "completes. (PCI Spec. Appendix C, Rule 8 d)", 0UL, 1UL, 0UL, NULL, 6UL,
        (b_int32) B_R_IRDY_4,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "DEVSEL 0", "DEVSEL 0:DEVSEL# must not be asserted during special "
        "cycle or if a reserved command was used. (PCI Spec. "
        "Sect. 3.7.2. Special Cycle and Sect. 3.1.1. "
        "Command Definition)", 0UL, 1UL, 0UL, NULL, 7UL,
        (b_int32) B_R_DEVSEL_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "DEVSEL 1", "DEVSEL 1:DEVSEL# must not be asserted when FRAME# "
        "is high or was sampled high on last clock edge. "
        "(for DAC DEVSEL# must be delayed by one cycle) "
        "(PCI Spec. Sect.3.3.1. Read Transaction - see "
        "timing diagram)", 0UL, 1UL, 0UL, NULL, 8UL,
        (b_int32) B_R_DEVSEL_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "DEVSEL 2", "DEVSEL 2:Once DEVSEL# has been asserted, it cannot "
        "be deasserted until the last data phase has completed, "
        "except to signal target-abort. (PCI Spec. "
        "Appendix C, Rule 35)", 0UL, 1UL, 0UL, NULL, 9UL,
        (b_int32) B_R_DEVSEL_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "DEVSEL 3", "DEVSEL 3:If not already deasserted, TRDY#, STOP# "
        "and DEVSEL# must be deasserted the clock following "
        "the completion of the last data phase and must be "
        "tri-stated the next clock. (PCI Spec. Appendix C, "
        "Rule 12 f)", 0UL, 1UL, 0UL, NULL, 10UL,
        (b_int32) B_R_DEVSEL_3,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TRDY 0", "TRDY 0:A target must assert DEVSEL# (claim the "
        "transaction) before it is allowed to issue any other "
        "target response. (TRDY#). (PCI Spec. Appendix C, "
        "Rule 34)", 0UL, 1UL, 0UL, NULL, 11UL,
        (b_int32) B_R_TRDY_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TRDY 1", "TRDY 1:TRDY# must not go low the first clock after "
        "assertion of FRAME# in a read transaction (a "
        "turnaround cycle is necessary). (PCI Spec. Sect. "
        "3.3.1. Read Transaction - see timing diagram)", 0UL, 1UL, 0UL, NULL, 12UL,
        (b_int32) B_R_TRDY_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TRDY 2", "TRDY 2:Once a target has asserted TRDY#, it cannot "
        "change DEVSEL#, TRDY#, STOP# until current data "
        "phase completes. (PCI Spec. Appendix C, Rule 12 d)", 0UL, 1UL, 0UL, NULL, 13UL,
        (b_int32) B_R_TRDY_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "STOP 0", "STOP 0:DEVSEL# must be asserted with or prior to "
        "the edge at which the target enables its outputs "
        "(STOP#). (PCI Spec. Appendix C, Rule 34)", 0UL, 1UL, 0UL, NULL, 14UL,
        (b_int32) B_R_STOP_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "STOP 1", "STOP 1:Once asserted, STOP# must remain asserted "
        "until FRAME# is deasserted; whereupon, STOP# must "
        "be deasserted. (PCI Spec. Appendix C, Rule 12 c)", 0UL, 1UL, 0UL, NULL, 15UL,
        (b_int32) B_R_STOP_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "STOP 2", "STOP 2:Once a target has asserted STOP#, it cannot "
        "change DEVSEL#, TRDY#, STOP# until current data phase "
        "completes. (PCI Spec. Appendix C, Rule 12 d)", 0UL, 1UL, 0UL, NULL, 16UL,
        (b_int32) B_R_STOP_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "LOCK 0", "LOCK 0:LOCK# must be asserted the clock following "
        "the (first) address phase and kept asserted to "
        "maintain control. (PCI Spec. Appendix C, Rule 32 e)", 0UL, 1UL, 0UL, NULL, 17UL,
        (b_int32) B_R_LOCK_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "LOCK 1", "LOCK 1:The first transaction of a locked access "
        "must be a READ. (PCI Spec. Appendix C, Rule 32 d)", 0UL, 1UL, 0UL, NULL, 18UL,
        (b_int32) B_R_LOCK_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "LOCK 2", "LOCK 2:LOCK# must be released if RETRY is signaled "
        "before a data phase has completed and lock has not "
        "been established or whenever an access is terminated "
        "by target-abort or master-abort. (PCI Spec. "
        "Appendix C, Rules 32 f and 32 g)", 0UL, 1UL, 0UL, NULL, 19UL,
        (b_int32) B_R_LOCK_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "CACHE 0", "CACHE 0:After HITM, CLEAN must be signaled before "
        "STANDBY (PCI Spec. Sect. 3.9.2. Cache State "
        "Transitions)", 0UL, 1UL, 0UL, NULL, 20UL,
        (b_int32) B_R_CACHE_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "CACHE 1", "CACHE 1:HITM must only be signaled after STANDBY "
        "(PCI Spec. Sect. 3.9.2. Cache State Transitions)", 0UL, 1UL, 0UL, NULL, 21UL,
        (b_int32) B_R_CACHE_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "PARITY 0", "PARITY 0:PERR# may never be asserted two clocks "
        "after address phase (or earlier in a transaction) or "
        "during a special cycle. During WRITE, PERR# may be "
        "asserted two clocks after IRDY#, during READ, PERR# "
        "may be asserted two clocks after TRDY#. "
        "(PCI Spec. Sect. 3.8.2. Error Reporting)", 0UL, 1UL, 0UL, NULL, 22UL,
        (b_int32) B_R_PARITY_0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "PARITY 1", "PARITY 1:AD[31 :: 0] address parity error "
        "(PCI Spec. Appendix C, Rule 37 b)", 0UL, 1UL, 0UL, NULL, 23UL,
        (b_int32) B_R_PARITY_1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "PARITY 2", "PARITY 2:AD [31 :: 0] data parity error has "
        "occured but was not signaled. (PCI Spec. "
        "Appendix C, Rule 37 b)", 0UL, 1UL, 0UL, NULL, 24UL,
        (b_int32) B_R_PARITY_2,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ARB 0", "ARB 0:When the current transaction is "
          "terminated by the target either by Retry or "
          "Disconnect the master must deassert its REQ# signal "
          "for a minimum of two clocks. (PCI Spec. Appendix C, Rule 10)",
          0UL, 1UL, 0UL, NULL, 36UL,
          (b_int32) B_R_ARB_0,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ARB 1", "ARB 1:One GNT# can be deasserted coincident with "
          "another GNT# being asserted only if the bus is not "
          "in idle state. (PCI Spec. Appendix C, Rule 23 b)",
          0UL, 1UL, 0UL, NULL, 37UL,
          (b_int32) B_R_ARB_1,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "ARB 2", "ARB 2:Only one GNT# must be asserted on any "
          "rising clock. (PCI Spec. Sect. 3.4. Arbitration)",
          0UL, 1UL, 0UL, NULL, 38UL,
          (b_int32) B_R_ARB_2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "PARITY 3", "PARITY 3:AD[31 :: 0] data parity error has "
          "occured (PCI Spec. Appendix C, Rule 37 b)",
          0UL, 1UL, 0UL, NULL, 39UL,
          (b_int32) B_R_PARITY_3,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "PARITY 4", "PARITY 4:AD[63 :: 32] address parity error has "
          "occured (PCI Spec. Appendix C, Rule 37 c)",
          0UL, 1UL, 0UL, NULL, 40UL,
          (b_int32) B_R_PARITY_4,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 0", "SEM 0:The earliest a master can terminate a "
          "transaction with master- abort is 4 clocks after "
          "entire address phase. (PCI Spec. Sect. 3.3.3.1. "
          "Master Initiated Termination and Sect. 3.10.1. "
          "64-bit Addressing on PCI)",
          0UL, 1UL, 0UL, NULL, 43UL,
          (b_int32) (b_int32) B_R_SEM_0,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 1", "SEM 1:Only certain combinations of AD[1:0] and "
          "C/BE#[3:0] are allowed during IO transfers. "
          "(PCI Spec. Sect. 3.2.2. Addressing)",
          0UL, 1UL, 0UL, NULL, 44UL,
          (b_int32) (b_int32) B_R_SEM_1,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 3", "SEM 3:Reserved commands are reserved for future "
          "use. (PCI Spec. Sect. 3.1.1. Command Definition)",
          0UL, 1UL, 0UL, NULL, 46UL,
          (b_int32) (b_int32) B_R_SEM_3,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 5", "SEM 5:A master must keep the byte enables stable "
          "during the complete data phase. (PCI Spec. Appendix C, Rule 3 b)",
          0UL, 1UL, 0UL, NULL, 48UL,
          (b_int32) (b_int32) B_R_SEM_5,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 6", "SEM 6:An INTx signal has been asserted and "
          "deasserted before an Interrupt Acknowledge cycle occurred.",
          0UL, 1UL, 0UL, NULL, 49UL,
          (b_int32) (b_int32) B_R_SEM_6,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 7", "SEM 7:Targets must signal Disconnect with or "
          "after the completion of the first data phase when "
          "a reserved burst mode is used during memory command "
          "transactions. (PCI Spec. Sect. 3.2.2 Addressing) ",
          0UL, 1UL, 0UL, NULL, 50UL,
          (b_int32) (b_int32) B_R_SEM_7,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      /* Rules SEM8 and SEM9 are disabled per default. These defaults are
       * currently set in: 1. dynamic.c (see below) 2. proplist.cxx 3.
       * setting.c 4. es_obs.c */
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 8", "SEM 8:A delayed transaction (retries) wasn't "
          "repeated within 2^14 clocks. (Agilent rule to detect "
          "potential deadlocks)",
          0x0UL, 0x1UL, 0x1UL, NULL, 51UL,
          (b_int32) (b_int32) B_R_SEM_8,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 9", "SEM 9:A delayed transaction hasn't terminated "
          "within 2^15 clocks. (Agilent rule to detect potential "
          "deadlocks)",
          0x0UL, 0x1UL, 0x1UL, NULL, 52UL,
          (b_int32) (b_int32) B_R_SEM_9,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 12", "SEM 12:The Memory Write and Invalidate command "
          "can only use the linear incrementing burst mode. "
          "(PCI Spec. 2.1, 3.2.2 Addressing, p.28)",
          0UL, 1UL, 0UL, NULL, 55UL,
          (b_int32) B_R_SEM_12,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "SEM 13", "SEM 13:When using the Memory write and invalidate "
          "command the transaction must begin at the start "
          "of a cacheline. (PCI Spec. 2.1, 3.2.2 Addressing)",
          0UL, 1UL, 0UL, NULL, 56UL,
          (b_int32) B_R_SEM_13,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "LAT 0", "LAT 0:Targets are required to complete the "
          "initial data phase of a transaction within 16 "
          "clocks, subsequent data phases within 8 clocks. "
          "(PCI Spec. Appendix C, Rules 25 and 26)",
          0UL, 1UL, 0UL, NULL, 57UL,
          (b_int32) B_R_LAT_0,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "LAT 1", "LAT 1:A master must complete all cycles within "
          "8 clocks. (PCI Spec. Appendix C, Rules 27)",
          0UL, 1UL, 0UL, NULL, 58UL,
          (b_int32) B_R_LAT_1,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));


      /* ---------------------  64 bit extension -------------------------- */
      /* ---------------------  64 bit extension -------------------------- */
      if (lic & B_CAPABILITY_64_BIT)  /* you need 64 Bit license for this */
      {
        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "W64 0", "W64 0:REQ64# mirrors FRAME## in a 64 bit "
            "transaction. REQ64# may only be asserted when "
            "FRAME# is asserted. Once asserted, it must stay "
            "low as long as FRAME#. (PCI Spec. Sect. 3.10. 64 "
            "Bit Bus Extension)",
            0UL, 1UL, 0UL, NULL, 32UL,
            (b_int32) B_R_W64_0,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "W64 1", "W64 1:ACK64# mirrors DEVSEL# in a "
            "64 bit transaction. ACK64# may only be asserted "
            "when DEVSEL# is asserted. Once asserted, it must "
            "stay low as long as DEVSEL#. (PCI Spec. Sect. "
            "3.10. 64-Bit Bus Extension)",
            0UL, 1UL, 0UL, NULL, 33UL,
            (b_int32) B_R_W64_1,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "W64 2", "W64 2:REQ64# must not be used with special "
            "cycle or interrupt acknowledge command. Only "
            "memory commands support 64 bit data transfers ("
            "no config commands, no IO commands). (PCI Spec. "
            "Sect. 3.10. 64-Bit Bus Extension)",
            0UL, 1UL, 0UL, NULL, 34UL,
            (b_int32) B_R_W64_2,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "W64 3", "W64 3:ACK64# may only be asserted, when REQ64# "
            "was asserted before (ACK64 is a response to REQ64#). "
            "(PCI Spec. Sect. 3.10. 64-Bit Bus Extension)",
            0UL, 1UL, 0UL, NULL, 35UL,
            (b_int32) B_R_W64_3,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "PARITY 5", "PARITY 5:AD[63 :: 32] data parity error has "
            "occured but was not signaled (PCI Spec. Appendix C, Rule 37 c)",
            0UL, 1UL, 0UL, NULL, 41UL,
            (b_int32) B_R_PARITY_5,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "PARITY 6", "PARITY 6:AD[63 :: 32] data parity error "
            "has occured (PCI Spec. Appendix C, Rule 37 c)",
            0UL, 1UL, 0UL, NULL, 42UL,
            (b_int32) B_R_PARITY_6,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "SEM 2", "SEM 2:A master that supports 64-bit addressing "
            "must generate a SAC instead of a DAC, when the "
            "upper 32 bits of the address are zero "
            "(PCI Spec. Sect. 3.10.1. 64-bit addressing on PCI)",
            0UL, 1UL, 0UL, NULL, 45UL,
            (b_int32) B_R_SEM_2,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "SEM 4", "SEM 4:DAC is not allowed immediately after a DAC.",
            0UL, 1UL, 0UL, NULL, 47UL,
            (b_int32) B_R_SEM_4,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "SEM 10", "SEM 10:During a Dual Address Cycle a 64 bit "
            "master has to drive the hi address on AD[63::32] "
            "in the initial and the second address phase ",
            0UL, 1UL, 0UL, NULL, 53UL,
            (b_int32) B_R_SEM_10,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "SEM 11", "SEM 11:During a Dual Address Cycle a 64 bit "
            "master has to drive the bus command on C/BE[7::4]# "
            "in the initial and the second address phase",
            0UL, 1UL, 0UL, NULL, 54UL,
            (b_int32) B_R_SEM_11,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

        B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
            "IRDY 5", "IRDY 5:IRDY# must not be asserted during the "
            "second phase of a DAC. (PCI Spec. Sect. 3.10.1 "
            "64-bit Addressing on PCI)",
            0UL, 1UL, 0UL, NULL, 59UL,
            (b_int32) B_R_IRDY_5,
            NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
      }
    }

  }

  /* associate the right number of elements */
  pObserverRulesGen->num_elem = num_elem;

  pCapable->observerrules_param = p_anchor; /* set it */

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the observer status information (generic and the five props ) */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pObserverStatusGen = pCapable->observerstatus_gen = (b_generic_infotype *) pVoid;

  /* leave depth of the observer rules at 0 (what else would make sense?) */
  pObserverStatusGen->group_name = "Observer Status";

  /* the observerrules rules information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "firsterr", "",
        0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_OBS_FIRSTERR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "accuerr", "",
        0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_OBS_ACCUERR,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "obsstat", "",
        0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_OBS_OBSSTAT,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "firsterr2", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_OBS_FIRSTERR2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "accuerr2", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_OBS_ACCUERR2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    }
  }

  /* associate the right number of elements */
  pObserverStatusGen->num_elem = num_elem;
  pCapable->observerstatus_param = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the timimg checker information (generic and the signals itself) */
  /* the paramrule information holds the info for the bit position ! */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));
  pCapable->timchecksig_gen = (b_generic_infotype *) pVoid;
  pCapable->timchecksig_gen->group_name = "Timing Checker signals";
  
  if (lic & B_CAPABILITY_ANALYZER && fHasIchiban)
  {
    if (theBoard == B_HW_E2925B  /* special case, supporting E2925B */
	|| theBoard == B_HW_E2925B_DEEP) /* including deep trace version */
    {
      pCapable->timchecksig_param=e2925b_tc_signal_table;
    }
    else
    {
      pCapable->timchecksig_param=e2926a_tc_signal_table;
    }
    num_elem=0;
    while (pCapable->timchecksig_param[num_elem++].param_name);
    pCapable->timchecksig_gen->num_elem=num_elem-1; /* 31 */
    /*assert(pCapable->timchecksig_gen->num_elem==31);*/
  }
  else
  {
    pCapable->timchecksig_param=NULL;
    pCapable->timchecksig_gen->num_elem=0;  
  }

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the timing checker status information (generic and props ) */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCapable->timcheckstatus_gen = (b_generic_infotype *) pVoid;

  pCapable->timcheckstatus_gen->group_name = "Timing Checker Status";

  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER && fHasIchiban)
  {

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "accusig", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_TC_ACCUSIG1,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "accusig2", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_TC_ACCUSIG2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "accusig3", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_TC_ACCUSIG3,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "accusig4", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_TC_ACCUSIG4,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "status", "",
          0UL, 1UL, 0UL, NULL, 0UL, (b_int32) B_TC_TCSTAT,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pCapable->timcheckstatus_gen->num_elem = num_elem;
  pCapable->timcheckstatus_param = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the timing checker properties information (generic and props ) */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCapable->timcheckprop_gen = (b_generic_infotype *) pVoid;

  pCapable->timcheckprop_gen->group_name = "Timing Checker Properties";

  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER && fHasIchiban)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "B_TC_HSIGN", "hsign",
          0UL,0x1UL,0UL, NULL, 0UL, (b_int32) B_TC_HSIGN,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "B_TC_SETUP_TIME", "setup",
          1000UL,9000UL,7000UL, NULL, 0UL, (b_int32) B_TC_SETUP_TIME,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "B_TC_HOLD_TIME", "hold",
          0UL,2000UL, 0UL, NULL, 0UL, (b_int32) B_TC_HOLD_TIME,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "B_TC_FREQUENCY", "freq",
          0UL, 67000000UL, 0UL, NULL, 0UL, (b_int32) B_TC_FREQUENCY,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pCapable->timcheckprop_gen->num_elem = num_elem;
  pCapable->timcheckprop_param = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */
  /* the timing checker generic properties information (generic and props ) */
  /* -------------------------------------------------------------------- */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCapable->timcheckgenprop_gen = (b_generic_infotype *) pVoid;

  pCapable->timcheckgenprop_gen->group_name = "Timing Checker Generic Properties";

  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER && fHasIchiban)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "B_TCGEN_SPEC", "spec",
          0UL,0x1UL,1UL, NULL, 0UL, (b_int32) B_TCGEN_SPEC,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no, 0UL));
  }

  /* associate the right number of elements */
  pCapable->timcheckgenprop_gen->num_elem = num_elem;
  pCapable->timcheckgenprop_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* the master attr group  string information (only strings ) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pMAttrGroupInfoGen = pCapable->mattrgroupinfo_gen = (b_generic_infotype *) pVoid;

  pMAttrGroupInfoGen->group_name = "Master Attribute Group Info";

  /* for group info, num_pages holds information if groups are supported by
   * hardware (== 1) or not (==0). Mainly used for PPR Programming & Reports */
  if (fHasIchiban)
    pMAttrGroupInfoGen->num_pages = 1;

  /* the triggerseq generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "MasterAddress0", "MA0", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_MATTR_GRP_MA0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_addr));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "MasterData0", "MD0", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_MATTR_GRP_MD0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_data));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "MasterData1", "MD1", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_MATTR_GRP_MD1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_data));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "MasterLast", "ML", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_MATTR_GRP_ML,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_last));

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "MasterAddress1", "MA1", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_MATTR_GRP_MA1,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_addr));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "MasterAddress2", "MA2", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_MATTR_GRP_MA2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_addr));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "MasterAddress3", "MA3", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_MATTR_GRP_MA3,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_addr));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "MasterAddress4", "MA4", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_MATTR_GRP_MA4,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_addr));

      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "MasterData2", "MD2", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_MATTR_GRP_MD2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_data));
    }
  }

  /* associate the right number of elements */
  pMAttrGroupInfoGen->num_elem = num_elem;

  pCapable->mattrgroupinfo_param = p_anchor;  /* set it */

  /* -------------------------------------------------------------------- */
  /* the target attr group  string information (only strings ) */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pTAttrGroupInfoGen = pCapable->tattrgroupinfo_gen = (b_generic_infotype *) pVoid;

  pTAttrGroupInfoGen->group_name = "Target Attribute Group Info";

  /* for group info, num_pages holds information if groups are supported by
   * hardware (== 1) or not (==0). Mainly used for PPR Programming & Reports */
  if (fHasIchiban)
    pTAttrGroupInfoGen->num_pages = 1;

  /* the target attr generic parameter information */
  p_anchor = NULL;              /* set to NULL to make AddParam Fct start
                                 * with malloc */
  num_elem = 0;

  if (lic & B_CAPABILITY_ANALYZER)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TargetAddress", "TA0", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_TATTR_GRP_TA0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_addr));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TargetData0", "TD0", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_TATTR_GRP_TD0,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_data));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TargetData1", "TD1", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_TATTR_GRP_TD1,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_data));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "TargetControl", "TC", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
        (b_int32) B_TATTR_GRP_TC,
        NULL, par_no, par_no, par_no, par_no, par_no, par_no,
        (b_int32) attr_grp_ctrl));

    if (fHasIchiban)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
          "TargetData2", "TD2", 0x0UL, 0x0UL, 0UL, NULL, 0UL,
          (b_int32) B_TATTR_GRP_TD2,
          NULL, par_no, par_no, par_no, par_no, par_no, par_no,
          (b_int32) attr_grp_data));
    }
  }

  /* associate the right number of elements */
  pTAttrGroupInfoGen->num_elem = num_elem;

  pCapable->tattrgroupinfo_param = p_anchor;  /* set it */


  /* -------------------------------------------------------------------- */
  /* cpu port interface information */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCPUportGen = pCapable->cpuport_gen = (b_generic_infotype *) pVoid;

  pCPUportGen->group_name = "CPU Port Interface"; /* group name */

  p_anchor = NULL;
  num_elem = 0;

  /* cpuport is unavailable on Compact PCI boards */
  if ( !fIsCompact && (lic & B_CAPABILITY_HOSTINT) )
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "B_CPU_MODE", "mode",
        0x0UL,                  /* min */
        0x2UL,                  /* max */
        0x2UL,                  /* default */
        NULL, (b_int32) B_PARAMRULE_MINMAXONLY,
        (b_int32) B_CPU_MODE, PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "B_CPU_PROTOCOL", "proto",
        0x0UL,                  /* min */
        0x0UL,                  /* max */
        0x0UL,                  /* default */
        NULL, 0UL, (b_int32) B_CPU_PROTOCOL,
        PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "B_CPU_RDYTYPE", "rdy",
        0x0UL,                  /* min */
        0x01UL,                 /* max */
        0x01UL,                 /* default */
        NULL, 0UL, (b_int32) B_CPU_RDYTYPE,
        PROPERTY_ENTRY));
  }

  pCPUportGen->num_elem = num_elem;
  pCapable->cpuport_param = p_anchor;

  /* -------------------------------------------------------------------- */
  /* static io interface information */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pStaticIOGen = pCapable->staticio_gen = (b_generic_infotype *) pVoid;

  pStaticIOGen->group_name = "Static IO Interface"; /* group name */

  if (!fIsCompact)
  {
    pStaticIOGen->depth = 8;      /* the number of static io pins we have */
  }
  else
  {
    pStaticIOGen->depth = 0;	/* no static IO pins on Compact PCI board */
  }

  p_anchor = NULL;
  num_elem = 0;

  if (!fIsCompact)
  {
    if (lic & B_CAPABILITY_HOSTINT)
    {
      B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			      "B_STAT_PINMODE", "pinmode",
			      (b_int32) B_PMD_INPONLY,/* min */
			      (b_int32) B_PMD_OPENDRAIN,  /* max */
			      (b_int32) B_PMD_OPENDRAIN,  /* default */
			      NULL, 0UL, (b_int32) B_STAT_PINMODE,
			      PROPERTY_ENTRY));
    }
  }

  pStaticIOGen->num_elem = num_elem;
  pCapable->staticio_param = p_anchor;

  /* -------------------------------------------------------------------- */
  /* power up properties */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pPuGen = pCapable->pu_gen = (b_generic_infotype *) pVoid;

  pPuGen->group_name = "Power Up";  /* group name */

  p_anchor = NULL;
  num_elem = 0;

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_PU_TRCRUNMODE", "trcrunmode",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_PU_TRCRUNMODE,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_PU_OBSRUNMODE", "obsrunmode",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_PU_OBSRUNMODE,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_PU_TCRUNMODE", "tcrunmode",
      0x0UL,                    /* min */
      0x02UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_PU_TCRUNMODE,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_PU_SSTRUNMODE", "sstrunmode",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_PU_SSTRUNMODE,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_PU_CONFRESTORE", "confrestore",
      (b_int32) B_CFRST_BIOSPU, /* min */
      (b_int32) B_CFRST_BIOS,   /* max */
      (b_int32) B_CFRST_BIOSPU, /* default */
      NULL, 0UL, (b_int32) B_PU_CONFRESTORE,
      PUConfRestore_table,
      par_no, par_no, par_no, par_no, par_no, par_no, 0UL));

  if (fHasIchiban)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "B_PU_MASTERRUNMODE", "mrunmode",
        0x0UL,                  /* min */
        (b_int32) B_MRUN_OFF,   /* max (one more than legal pages) */
        (b_int32) B_MRUN_OFF,   /* default */
        NULL, 0UL, (b_int32) B_PU_MASTERRUNMODE,
        PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
        "B_PU_TCRUNMODE", "tcrunmode",
        0x0UL,                  /* min */
        0x2UL,                  /* max  */
        0x0UL,   /* default */
        NULL, 0UL, (b_int32) B_PU_TCRUNMODE,
        PROPERTY_ENTRY));
  }

  pPuGen->num_elem = num_elem;
  pCapable->pu_param = p_anchor;

  /* -------------------------------------------------------------------- */
  /* board properties */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pBoardGen = pCapable->board_gen = (b_generic_infotype *) pVoid;

  pBoardGen->group_name = "Board";  /* group name */

  p_anchor = NULL;
  num_elem = 0;

  if (fHasIchiban)
  {
    zwval = B_RSTMODE_RESETCONFIG;
    zwvaldef = B_RSTMODE_RESETCONFIG;
  }
  else
  {
    zwval = B_RSTMODE_RESETSM;
    zwvaldef = B_RSTMODE_RESETALL;
  }

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_BOARD_RSTMODE", "rstmode",
      (b_int32) B_RSTMODE_RESETALL, /* min */
      zwval,                    /* max */
      zwvaldef,                 /* default */
      NULL, 0UL, (b_int32) B_BOARD_RSTMODE,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_BOARD_PERREN", "perren",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_BOARD_PERREN,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_BOARD_SERREN", "serren",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_BOARD_SERREN,
      PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
      "B_BOARD_ROMUSAGE", "romusage",
      0x0UL,                    /* min */
      0x01UL,                   /* max */
      0x0UL,                    /* default */
      NULL, 0UL, (b_int32) B_BOARD_ROMUSAGE,
      PROPERTY_ENTRY));

  if (fIsCompact)
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_BOARD_HOTSWAPMODE", "hsmode",
			    B_BOARD_HOTSWAP_NONE, 	/* min */
			    B_BOARD_HOTSWAP_STEALTH, 	/* max  */
			    B_BOARD_HOTSWAP_HIGH, 	/* default */
			    NULL, /* default String */
			    B_PARAMRULE_NORULE,
			    (b_int32) B_BOARD_HOTSWAPMODE,
			    BoardHSMode_table,
			    par_no, par_no, par_no, par_no, par_no, par_no,
			    0 ));
  }
  
  pBoardGen->num_elem = num_elem;
  pCapable->board_param = p_anchor;

  /* -------------------------------------------------------------------- */
  /* SST config properties */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSstCfgGen = pCapable->sstcfg_gen = (b_generic_infotype *) pVoid;

  pSstCfgGen->group_name = "SST Config Properties";  /* group name */

  p_anchor = NULL;
  num_elem = 0;

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTCFG_CTRL", "ctrl",
			  (b_int32)B_CTRL_PROG,			/* min */
			  (b_int32)B_CTRL_STANDALONE,     	/* max */
			  (b_int32)B_CTRL_PROG,                 /* default */
			  NULL, 0UL, (b_int32)B_SSTCFG_CTRL,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTCFG_SHELL", "shell",
			  (b_int32)B_SHELL_NOT_USED,   		/* min */
			  (b_int32)B_SHELL_USED,	     	/* max */
			  (b_int32)B_SHELL_NOT_USED,    	/* default */
			  NULL, 0UL, (b_int32)B_SSTCFG_SHELL,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTCFG_HOST", "host",
			  (b_int32)B_HOST_DUT,   		/* min */
			  (b_int32)B_HOST_EXTERNAL,	     	/* max */
			  (b_int32)B_HOST_DUT,	    		/* default */
			  NULL, 0UL, (b_int32)B_SSTCFG_HOST,
			  PROPERTY_ENTRY));

  pSstCfgGen->num_elem = num_elem;
  pCapable->sstcfg_param = p_anchor;
  
  /* -------------------------------------------------------------------- */
  /* SST properties */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pSstGen = pCapable->sst_gen = (b_generic_infotype *) pVoid;

  pSstGen->group_name = "SST Properties";  /* group name */

  p_anchor = NULL;
  num_elem = 0;

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_APAGE1", "apage1",
			  0,					/* min */
			  63,     			    /* max */
			  1,                    /* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_APAGE1,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_AUTOMASK", "automask",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  5,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_AUTOMASK,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_BUSUTIL", "busutil",
			  0,					/* min */
			  0x100,     			/* max */
			  30,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_BUSUTIL,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_CONFIGSCAN", "configscan",
			  B_CONFIGSCAN_OFF,    	       		/* min */
			  B_CONFIGSCAN_AT_ERROR,               	/* max */
			  B_CONFIGSCAN_AT_BEGIN,       		/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_CONFIGSCAN,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_DATAPATT", "datapatt",
			  B_DATAPATT_CONSTANT,			/* min */
			  B_DATAPATT_WALKINGZERO,     		/* max */
			  B_DATAPATT_CONSTANT,                 	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_DATAPATT,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_DATAVAL1", "dataval1",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  0,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_DATAVAL1,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_DATAVAL2", "dataval2",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  0,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_DATAVAL2,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_DURATION", "duration",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  600,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_DURATION,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_EVENT", "event",
			  B_EVENT_RUN,		       		/* min */
			  B_EVENT_PAUSE,     			/* max */
			  B_EVENT_ABORT,       			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_EVENT,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_HEARTBEATVALUE", "hbvalue",
			  5120,		      	       		/* min */
			  16383,	     			/* max */
			  5120,		 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_HEARTBEATVALUE,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_ID", "id",
			  B_SSTID_INTERN,      	       		/* min */
			  B_SSTID_LAST,     			/* max */
			  B_SSTID_PROTCHECK, 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_ID,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_INTADDR", "iad",
			  0,      	       		        /* min */
			  0x080000,     			/* max */
			  0x000000, 			        /* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_INTADDR,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_OBSERVER", "observer",
			  B_OBSERVER_OFF,      	       		/* min */
			  B_OBSERVER_ON,     			/* max */
			  B_OBSERVER_ON, 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_OBSERVER,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFCTR1", "perfctr1",
			  0,          		                /* min */
			  0xffffffff,            		/* max */
			  0,			                /* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFCTR1,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFCTR2", "perfctr2",
			  0,          		                /* min */
			  0xffffffff,            		/* max */
			  0,			                /* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFCTR2,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFMODE", "perfmode",
			  B_PERFMODE_UTILIZATION,      		/* min */
			  B_PERFMODE_ASIS,       		/* max */
			  B_PERFMODE_UTILIZATION, 	       	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFMODE,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFORMANCE", "perf",
			  B_PERFORMANCE_OFF,          		/* min */
			  B_PERFORMANCE_ON,            		/* max */
			  B_PERFORMANCE_ON, 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFORMANCE,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFREPINTV", "perfrepintv",
			  0,			      		/* min */
			  0xffffffff,		       		/* max */
			  600,			 	       	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFREPINTV,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PERFREPORT", "perfreport",
			  B_PERFREPORT_OFF,      		/* min */
			  B_PERFREPORT_ALWAYS,       		/* max */
			  B_PERFREPORT_ON_INTERVAL, 	       	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PERFREPORT,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_PROT1", "prot1",
			  B_PROT_LITE,      		        /* min */
			  B_PROT_ASIS,       		        /* max */
			  B_PROT_HARD, 	       	                /* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_PROT1,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_QUERYINTERVAL", "queryinterval",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  10,                 			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_QUERYINTERVAL,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_TRACEMEM", "tracemem",
			  B_TRACEMEM_OFF,      			/* min */
			  B_TRACEMEM_ON,     			/* max */
			  B_TRACEMEM_ON,        			/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_TRACEMEM,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_TRACEDEPTH", "tracedepth",
			  0,					/* min */
			  0xffffffff,     			/* max */
			  0x1000,                 	       	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_TRACEDEPTH,
			  PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			  "B_SSTPAR_TRIGGER", "trigger",
			  B_TRIGGER_PROTVIOLATION,  	       	/* min */
			  B_TRIGGER_ASIS,	     	       	/* max */
			  B_TRIGGER_PROTVIOLATION,             	/* default */
			  NULL, 0UL, (b_int32)B_SSTPAR_TRIGGER,
			  PROPERTY_ENTRY));

/***********************/


  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTALARM", "alarm",
              1,                    /* min */
              0xffffffff,           /* max */
              10000000,             /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTALARM,
              PROPERTY_ENTRY));


  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTROUTE", "route",
              0,                    /* min */
              1,                    /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTROUTE,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTCANCEL", "cancel",
              0,                    /* min */
              2,                    /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTCANCEL,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTERRUPTMASK", "intr",
              0,                    /* min */
              15,                   /* max */
              1,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTERRUPTMASK,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTGAP_A", "gap_a",
              0,                    /* min */
              0xffffffff,           /* max */
              100000,               /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTGAP_A,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTGAP_B", "gap_b",
              0,                    /* min */
              0xffffffff,           /* max */
              100000,               /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTGAP_B,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTGAP_C", "gap_c",
              0,                    /* min */
              0xffffffff,           /* max */
              100000,               /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTGAP_C,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INTGAP_D", "gap_d",
              0,                    /* min */
              0xffffffff,           /* max */
              100000,               /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INTGAP_D,
              PROPERTY_ENTRY));


  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ISRDWELL_A", "dwell_a",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ISRDWELL_A,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ISRDWELL_B", "dwell_b",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ISRDWELL_B,
              PROPERTY_ENTRY));
			  
  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ISRDWELL_C", "dwell_c",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ISRDWELL_C,
              PROPERTY_ENTRY));
			  
  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ISRDWELL_D", "dwell_d",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ISRDWELL_D,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_STARTADDR1", "saddr1",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_STARTADDR1,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_STARTADDR2", "saddr2",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_STARTADDR2,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_RANGE1", "range1",
              0,                    /* min */
              0xffffffff,           /* max */
              0,  /* off */         /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_RANGE1,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_RANGE2", "range2",
              0,                    /* min */
              0xffffffff,           /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_RANGE2,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ACCESSMODE", "mode",
              0,                    /* min */
              3,                    /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ACCESSMODE,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_ADDRSEQ", "addrseq",
              0,                    /* min */
              1,                    /* max */
              0,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_ADDRSEQ,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_RPTCOUNT", "rptcount",
              1,                    /* min */
              0xffffffff,           /* max */
              1,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_RPTCOUNT,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_READCMD", "readcmd",
              0,                    /* min */
              7,                    /* max */
              1,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_READCMD,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_BLKSIZE", "blksize",
              1024,                 /* min */
              65536,                /* max */
              4096,                 /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_BLKSIZE,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_UTILLIMIT", "utillimit",
              10,                    /* min */
              95,                    /* max */
              95,                    /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_UTILLIMIT,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INT_OFFSET1", "isaddr1",
              0,                     /* min */
              0xffffffff,            /* max (Tracememory size) */
              0,                     /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INT_OFFSET1,
              PROPERTY_ENTRY));

  B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
              "B_SSTPAR_INT_OFFSET2", "isaddr2",
              0,                     /* min */
              0xffffffff,            /* max (Tracememory size) */
              0,                     /* default */
              NULL, 0UL, (b_int32)B_SSTPAR_INT_OFFSET2,
              PROPERTY_ENTRY));



/**********************/

  if (!fHasIchiban) {
  }
  
  pSstGen->num_elem = num_elem;
  pCapable->sst_param = p_anchor;
  
  /* -------------------------------------------------------------------- */
  /* compact pci pin interface information */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCPCIPinGen = pCapable->cpcipin_gen = (b_generic_infotype *) pVoid;

  pCPCIPinGen->group_name = "CPCI Pins"; /* group name */

  p_anchor = NULL;
  num_elem = 0;

  /* only available on Compact PCI boards */
  if ( fIsCompact && (lic & B_CAPABILITY_EXERCISER) )
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_BDSEL", "bdsel",
			    0x0UL,                  /* min */
			    0x1UL,                  /* max */
			    0x1UL,                  /* default */
			    NULL, (b_int32) B_PARAMRULE_MINMAXONLY,
			    (b_int32) B_CPCI_BDSEL, PROPERTY_ENTRY));
    
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_ENUM", "enum",
			    0x0UL,                  /* min */
			    0x1UL,                  /* max */
			    0x1UL,                  /* default */
			    NULL, 0UL,
			    (b_int32) B_CPCI_ENUM, PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_HEALTHY", "healthy",
			    0x0UL,                  /* min */
			    0x01UL,                 /* max */
			    0x01UL,                 /* default */
			    NULL, 0UL,
			    (b_int32) B_CPCI_HEALTHY, PROPERTY_ENTRY));
  }

  pCPCIPinGen->num_elem = num_elem;
  pCapable->cpcipin_param = p_anchor;

  /* -------------------------------------------------------------------- */
  /* compact pci status information */
  /* -------------------------------------------------------------------- */

  /* malloc, chk ptr and zero out new mem */
  B_ERRCHECK(malloc_chk_zero(&pVoid, sizeof(b_generic_infotype)));

  /* short cut */
  pCPCIStatusGen = pCapable->cpcistatus_gen = (b_generic_infotype *) pVoid;

  pCPCIStatusGen->group_name = "CPCI Status"; /* group name */

  p_anchor = NULL;
  num_elem = 0;

  /* only available on Compact PCI boards */
  if ( fIsCompact && (lic & B_CAPABILITY_EXERCISER) )
  {
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_INS", "ins",
			    0x0UL,                  /* min */
			    0x1UL,                  /* max */
			    0x0UL,                  /* default */
			    NULL, (b_int32) B_PARAMRULE_MINMAXONLY,
			    (b_int32) B_CPCI_INS, PROPERTY_ENTRY));
    
    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_EXT", "ext",
			    0x0UL,                  /* min */
			    0x1UL,                  /* max */
			    0x0UL,                  /* default */
			    NULL, 0UL,
			    (b_int32) B_CPCI_EXT, PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_LOO", "loo",
			    0x0UL,                  /* min */
			    0x01UL,                 /* max */
			    0x00UL,                 /* default */
			    NULL, 0UL,
			    (b_int32) B_CPCI_LOO, PROPERTY_ENTRY));

    B_ERRCHECK(BestAddParam(&p_anchor, num_elem++,
			    "B_CPCI_EIM", "eim",
			    0x0UL,                  /* min */
			    0x01UL,                 /* max */
			    0x00UL,                 /* default */
			    NULL, 0UL,
			    (b_int32) B_CPCI_EIM, PROPERTY_ENTRY));
  }

  pCPCIStatusGen->num_elem = num_elem;
  pCapable->cpcistatus_param = p_anchor;

  
  B_ERRETURN(B_E_OK);
}


/* that function is so bloated that we need a new section ! */
#if !defined(_WIN32) && (defined(_DOS) || defined(DOS))
#pragma code_seg("dyn_code2","CODE")
#endif /* dos */

/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------ */

/* ------------------------------------------------------------------------
 * This function deletes the dynamic capabilities and frees all allocated
 * memory.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestCapaClose(b_handletype handle)
{
  b_dynamic_capability_type *pCapable = handle_array[handle].capable;
  if (pCapable == NULL)
    B_ERRETURN(B_E_OK);

  free(pCapable->capa_code);

  free(pCapable->masterblock_gen);
  free(pCapable->masterblock_param);

  free(pCapable->masterattr_gen);
  free(pCapable->masterattr_param);

  free(pCapable->mastergen_gen);
  free(pCapable->mastergen_param);

  free(pCapable->exercisergen_gen);
  free(pCapable->exercisergen_param);

  free(pCapable->targetattr_gen);
  free(pCapable->targetattr_param);

  free(pCapable->targetgen_gen);
  free(pCapable->targetgen_param);

  free(pCapable->decoder_std_gen);
  free(pCapable->decoder_std_param);

  free(pCapable->decoder_gen);
  free(pCapable->decoder_param);

  free(pCapable->decoder_exp_gen);
  free(pCapable->decoder_exp_param);

  free(pCapable->decoder_cfg0_gen);
  free(pCapable->decoder_cfg0_param);

  free(pCapable->decoder_cfg_gen);
  free(pCapable->decoder_cfg_param);

  free(pCapable->decoder_sub_gen);
  free(pCapable->decoder_sub_param);

  free(pCapable->decoder_ucfg_gen);
  free(pCapable->decoder_ucfg_param);

  free(pCapable->tracegen_gen);
  free(pCapable->tracegen_param);

  free(pCapable->tracemem_gen);
  /* free(pCapable->tracemem_param); */
  /* not freed because it is static */

  free(pCapable->pattern_gen);

  if (!BestIs2925(handle))
    /* not freed for e2925 because it is static */
    free(pCapable->pattern_param);

  /* Observer */
  free(pCapable->observerrules_gen);
  free(pCapable->observerrules_param);
  free(pCapable->observerstatus_gen);
  free(pCapable->observerstatus_param);

  /* Timing Checker */
  free(pCapable->timcheckstatus_gen);
  free(pCapable->timcheckstatus_param);
  free(pCapable->timcheckprop_gen);
  free(pCapable->timcheckprop_param);
  free(pCapable->timcheckgenprop_gen);
  free(pCapable->timcheckgenprop_param);
  free(pCapable->timchecksig_gen);
  /* free(pCapable->timchecksig_param); this is a static array, so dont free it ! */

  free(pCapable->datamemory_gen);

  free(pCapable->seqperforpar_gen);
  free(pCapable->seqperforpar_param);

  free(pCapable->seqperforgen_gen);
  free(pCapable->seqperforgen_param);

  free(pCapable->seqtriggergen_gen);
  free(pCapable->seqtriggergen_param);

  free(pCapable->seqtriggerpar_gen);
  free(pCapable->seqtriggerpar_param);

  free(pCapable->mattrgroupinfo_gen);
  free(pCapable->mattrgroupinfo_param);

  free(pCapable->tattrgroupinfo_gen);
  free(pCapable->tattrgroupinfo_param);

  free(pCapable->cpuport_gen);
  free(pCapable->cpuport_param);

  free(pCapable->staticio_gen);
  free(pCapable->staticio_param);

  free(pCapable->pu_gen);
  free(pCapable->pu_param);

  free(pCapable->board_gen);
  free(pCapable->board_param);

  free(pCapable->cpcipin_gen);
  free(pCapable->cpcipin_param);

  free(pCapable->cpcistatus_gen);
  free(pCapable->cpcistatus_param);
  
  free(pCapable->sstcfg_gen);
  free(pCapable->sstcfg_param);

  free(pCapable->sst_gen);
  free(pCapable->sst_param);

  free(pCapable);

  /* prevent a second "free" */
  handle_array[handle].capable = NULL;

  B_ERRETURN(B_E_OK);
}


/* tries to access the param info structure. In essence checks for the
   existance of a parameter */
b_errtype EXPORT BestParamProbe (
    b_handletype        handle,
    b_paramtype         param_typ,   /* what kind of parameter at all */      
    b_int32             parnum
    )
{
  return BestParamInfoGet (handle, param_typ, parnum, NULL, B_ENUM_SEARCH);
}


/* ------------------------------------------------------------------------
 * This function checks a Parameter if the value is within the correct
 * range. The range depends on hardware and license information.
 * Example: ACK64# can only be set if hardware is Ichiban and license for
 * 64 bit exerciser is valid.
 * In this example only the setting to 1 might be illegal (if exerciser
 * itself is licensed but not the 64 bit option).
 * Illegal is always a setting of a parameter different than the default
 * value if there is no license available.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestParamCheck(
    b_handletype handle,
    b_paramtype param_typ,      /* kind of parameter */
    b_int32 prop_typ,           /* the parameter itself */
    b_int32 value)
{                               /* erweiterung um strings noetig !!! */
  b_errtype err;

  b_param_infotype *myinfo;
  b_generic_infotype *mygeninfo;


  B_ERRCHECK(BestGenInfoGet(handle, param_typ, &mygeninfo));

  BestLastErrorAllParamsSet(handle, (b_int32) param_typ, prop_typ, 0UL, 0UL, value);

  B_ERRCHECK(BestParamInfoGet(handle, param_typ, prop_typ, &myinfo,
    (b_int32) B_ENUM_SEARCH));


  if (myinfo->paramrule & (b_int32) B_PARAMRULE_BITSET)
  {
    /* e.g. decoder speed for Ichiban20 allows values 0,1 or 4 
     * RULE:
     *     If the i-th bit (0,..,31) of max_val is set, 
     *     the value min_val+i is allowed: 	    
     */
    /* SCR; bracketed...hope it's what you meant! */
    if ( (value > (myinfo->min_val + 31))
      || (value < myinfo->min_val)
      || !((1UL << (int)(value - myinfo->min_val)) & myinfo->max_val)
      )
	{
      BestLastErrorParamSet(handle, B_ERRPAR_3, value);
      B_ERRETURN(B_E_VALUE);
    }		 
    else
	{
      B_ERRETURN(B_E_OK);
    }
  }		

  if (myinfo->paramrule & (b_int32) B_PARAMRULE_ZERO2WIDTH)
  {
    /* e.g. Timing Checker (see e2926a_tc_signal_table[]) */
    /* allowed rage is from 0 to 2^width-1; width is stored in max_val */
    
    b_int32 MaxVal = 
      (myinfo->max_val == 32L ? 0xffffffffL : (1UL << (int)myinfo->max_val) - 1L);

    BestLastErrorAllParamsSet(handle, (b_int32) param_typ, prop_typ,
      0, MaxVal, value);

    /* now check if the parameter is within the bounds */
    if (value > MaxVal)
    {
      B_ERRETURN(B_E_RANGE);
    }
  }
  else
  {
    /* just in case we have an error, set the parameter so the error is
     * traceable and can get restored later */
    BestLastErrorAllParamsSet(handle, (b_int32) param_typ, prop_typ,
      myinfo->min_val, myinfo->max_val, value);

    /* now check if the parameter is within the bounds */
    if (((myinfo->max_val < value) || (myinfo->min_val > value)) &&
      !(value == 0 && (myinfo->paramrule & (b_int32) B_PARAMRULE_ZEROALLOWED)) &&
      !(value == 0xffffffffL &&
        (myinfo->paramrule & (b_int32) B_PARAMRULE_MAXLONGALLOWED)))
    {
      B_ERRETURN(B_E_RANGE);
    }

    /* now check if the parameter behaves according to the parameter rules */
    /* first check if the parameter is at a dword boundary */
    if ((param_typ != B_PARAM_OBSRULES) &&
      (myinfo->paramrule & (b_int32) B_PARAMRULE_DWORDBOUND) && (value & 0x0000003))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_4, value);
      B_ERRETURN(B_E_ALIGN);
    }

    if ((param_typ != B_PARAM_OBSRULES) &&
      (myinfo->paramrule & (b_int32) B_PARAMRULE_MINMAXONLY) &&
      (value != myinfo->min_val) &&
      (value != myinfo->max_val))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_3, value);
      B_ERRETURN(B_E_VALUE);
    }
  }
  
  B_ERRETURN(B_E_OK);
}




/* ------------------------------------------------------------------------
 * The compilers (application) and the GUI  need to know information about
 * the hardware and its capabilities. This information is seperated into
 * a generic info (that is the number of parameters or signals, the depth
 * of the corresponding memory and so on). This information is given here.
 * The calling function gets a generic_infotype struct returned and
 * depending on the typ of information requested the fields within in the
 * struct contain the corresponding information.
 * ------------------------------------------------------------------------ */
static b_errtype IBestGenInfoGet(
    b_handletype handle,
    b_paramtype param_typ,      /* what kind of param block at all */
    b_generic_infotype ** geninfo,  /* the info struct */
    b_int8 generate_errors      /* use this to indicate if you want error
                                 * info set */
)
{
  /* code size reduction ... note that this we can't trust this ptr yet! */
  b_dynamic_capability_type *pCapability = handle_array[handle].capable;
  
  if (generate_errors)
  {
    HANDLECHECK;
  }
  else
  {
    if (!((handle >= 0) && (handle < MAXHANDLES) &&
        handle_array[handle].is_open))
      return B_E_BAD_HANDLE;
  }

  /* now we can trust 'handle' and thus pCapability */
  
  if (pCapability == NULL) /* if no info available->Error */
  {
    if (generate_errors)
      B_ERRETURN(B_E_DYNAMIC_CAPABILITY);
    else
      return B_E_DYNAMIC_CAPABILITY;
  }

  switch (param_typ)
  {
  case B_PARAM_MASTER_BLOCK:
    *geninfo = pCapability->masterblock_gen;
    break;
  case B_PARAM_MASTER_ATTR:
    *geninfo = pCapability->masterattr_gen;
    break;
  case B_PARAM_MASTER_GEN:
    *geninfo = pCapability->mastergen_gen;
    break;
  case B_PARAM_EXERCISER_GEN:
    *geninfo = pCapability->exercisergen_gen;
    break;
  case B_PARAM_DECODER_STD:
    *geninfo = pCapability->decoder_std_gen;
    break;
  case B_PARAM_DECODER_GEN:
    *geninfo = pCapability->decoder_gen;
    break;
  case B_PARAM_DECODER_EXP:
    *geninfo = pCapability->decoder_exp_gen;
    break;
  case B_PARAM_DECODER_CFG0:
    *geninfo = pCapability->decoder_cfg0_gen;
    break;
  case B_PARAM_DECODER_CFG:
    *geninfo = pCapability->decoder_cfg_gen;
    break;
  case B_PARAM_DECODER_SUB:
    *geninfo = pCapability->decoder_sub_gen;
    break;
  case B_PARAM_DECODER_UCFG:
    *geninfo = pCapability->decoder_ucfg_gen;
    break;
  case B_PARAM_TARGET_ATTR:
    *geninfo = pCapability->targetattr_gen;
    break;
  case B_PARAM_TARGET_GEN:
    *geninfo = pCapability->targetgen_gen;
    break;
  case B_PARAM_TRACE_MEM:
    *geninfo = pCapability->tracemem_gen;
    break;
  case B_PARAM_TRACE_GEN:
    *geninfo = pCapability->tracegen_gen;
    break;
  case B_PARAM_PATTERN:
    *geninfo = pCapability->pattern_gen;
    break;
  case B_PARAM_DATA_MEM:
    *geninfo = pCapability->datamemory_gen;
    break;
  case B_PARAM_OBSRULES:
    *geninfo = pCapability->observerrules_gen;
    break;
  case B_PARAM_OBSSTATUS:
    *geninfo = pCapability->observerstatus_gen;
    break;
  case B_PARAM_TCSIG:
    *geninfo = pCapability->timchecksig_gen;
    break;
  case B_PARAM_TCSTATUS:
    *geninfo = pCapability->timcheckstatus_gen;
    break;
  case B_PARAM_TCPROP:
    *geninfo = pCapability->timcheckprop_gen;
    break;
  case B_PARAM_TCGENPROP:
    *geninfo = pCapability->timcheckgenprop_gen;
    break;
  case B_PARAM_MAGROUPINFO:
    *geninfo = pCapability->mattrgroupinfo_gen;
    break;
  case B_PARAM_TAGROUPINFO:
    *geninfo = pCapability->tattrgroupinfo_gen;
    break;
  case B_PARAM_CPUPORT:
    *geninfo = pCapability->cpuport_gen;
    break;
  case B_PARAM_STATICIO:
    *geninfo = pCapability->staticio_gen;
    break;
  case B_PARAM_PUINFO:
    *geninfo = pCapability->pu_gen;
    break;
  case B_PARAM_BOARD:
    *geninfo = pCapability->board_gen;
    break;
  case B_PARAM_SSTCFG:
    *geninfo = pCapability->sstcfg_gen;
    break;
  case B_PARAM_SSTPROP:
    *geninfo = pCapability->sst_gen;
	break;
  case B_PARAM_CPCI_STATUS:
    *geninfo = pCapability->cpcistatus_gen;
	break;
  case B_PARAM_CPCI_PIN:
    *geninfo = pCapability->cpcipin_gen;
    break;
  default:
    *geninfo = NULL;
    if (generate_errors)
      B_ERRETURN(B_E_DYNAMIC_CAPABILITY);
    else
      return B_E_DYNAMIC_CAPABILITY;
  }                             /*lint !e788 */
  if (generate_errors)
    B_ERRETURN(B_E_OK);
  return B_E_OK;
}
/* the officially exported function, calls internal get with parameter true
 * for the error generation */
b_errtype EXPORT BestGenInfoGet(
    b_handletype handle,
    b_paramtype param_typ,      /* what kind of param block at all */
    b_generic_infotype ** geninfo /* the info struct */
)
{
  B_ERRETURN(IBestGenInfoGet(handle, param_typ, geninfo, 1));
}
/* ------------------------------------------------------------------------
 * For the parameters itself this function needs to get called. Using an
 * Id to identify the parameters gives the compilers the ability to access
 * the information.
 * ------------------------------------------------------------------------ */
static b_errtype IBestParamInfoGet(
    b_handletype handle,
    b_paramtype param_typ,      /* what kind of parameter at all */
    b_int64 parnum,             /* the parameter itself (as a number) */
    b_param_infotype ** paraminfo,  /* the info struct */
    b_int32 option,             /* how to search for params (index or enum */
    b_int8 generate_errors      /* do or not do error handling */
)
{
  /* code size reduction ... note that this we can't trust this ptr yet! */
  b_dynamic_capability_type *pCapability = handle_array[handle].capable;
  
  b_param_infotype *myinfo;
  b_generic_infotype *mygeninfo;
  b_int64 id;
  
  if (generate_errors == DO_STANDARD_ERROR_HANDLING)
  {
    HANDLECHECK;
  }
  else
  {
    if (!__HANDLECHECK)
      return B_E_BAD_HANDLE;
  }


  if (generate_errors == DO_STANDARD_ERROR_HANDLING)
    BestLastErrorAllParamsSet(handle, (b_int32) param_typ, parnum, 0UL, 0UL, 0UL);

  /* now we can trust 'handle' and thus pCapability */
  
  if (pCapability == NULL) /* if no info available->Error */
  {
    if (generate_errors == DO_STANDARD_ERROR_HANDLING)
      B_ERRETURN(B_E_DYNAMIC_CAPABILITY);
    else
      return B_E_DYNAMIC_CAPABILITY;
  }

  switch (param_typ)
  {
  case B_PARAM_MASTER_BLOCK:
    myinfo = pCapability->masterblock_param;
    mygeninfo = pCapability->masterblock_gen;
    break;
  case B_PARAM_MASTER_ATTR:
    myinfo = pCapability->masterattr_param;
    mygeninfo = pCapability->masterattr_gen;
    break;
  case B_PARAM_MASTER_GEN:
    myinfo = pCapability->mastergen_param;
    mygeninfo = pCapability->mastergen_gen;
    break;
  case B_PARAM_EXERCISER_GEN:
    myinfo = pCapability->exercisergen_param;
    mygeninfo = pCapability->exercisergen_gen;
    break;
  case B_PARAM_DECODER_GEN:
    myinfo = pCapability->decoder_param;
    mygeninfo = pCapability->decoder_gen;
    break;
  case B_PARAM_DECODER_STD:
    myinfo = pCapability->decoder_std_param;
    mygeninfo = pCapability->decoder_std_gen;
    break;
  case B_PARAM_DECODER_EXP:
    myinfo = pCapability->decoder_exp_param;
    mygeninfo = pCapability->decoder_exp_gen;
    break;
  case B_PARAM_DECODER_CFG0:
    myinfo = pCapability->decoder_cfg0_param;
    mygeninfo = pCapability->decoder_cfg0_gen;
    break;
  case B_PARAM_DECODER_CFG:
    myinfo = pCapability->decoder_cfg_param;
    mygeninfo = pCapability->decoder_cfg_gen;
    break;
  case B_PARAM_DECODER_SUB:
    myinfo = pCapability->decoder_sub_param;
    mygeninfo = pCapability->decoder_sub_gen;
    break;
  case B_PARAM_DECODER_UCFG:
    myinfo = pCapability->decoder_ucfg_param;
    mygeninfo = pCapability->decoder_ucfg_gen;
    break;
  case B_PARAM_TARGET_ATTR:
    myinfo = pCapability->targetattr_param;
    mygeninfo = pCapability->targetattr_gen;
    break;
  case B_PARAM_TARGET_GEN:
    myinfo = pCapability->targetgen_param;
    mygeninfo = pCapability->targetgen_gen;
    break;
  case B_PARAM_TRACE_MEM:
    myinfo = pCapability->tracemem_param;
    mygeninfo = pCapability->tracemem_gen;
    break;
  case B_PARAM_TRACE_GEN:
    myinfo = pCapability->tracegen_param;
    mygeninfo = pCapability->tracegen_gen;
    break;
  case B_PARAM_PATTERN:
    myinfo = pCapability->pattern_param;
    mygeninfo = pCapability->pattern_gen;
    break;
  case B_PARAM_OBSRULES:
    mygeninfo = pCapability->observerrules_gen;
    myinfo = pCapability->observerrules_param;
    break;
  case B_PARAM_OBSSTATUS:
    mygeninfo = pCapability->observerstatus_gen;
    myinfo = pCapability->observerstatus_param;
    break;
  case B_PARAM_TCSIG:
    mygeninfo = pCapability->timchecksig_gen;
    myinfo = pCapability->timchecksig_param;
    break;
  case B_PARAM_TCSTATUS:
    mygeninfo = pCapability->timcheckstatus_gen;
    myinfo = pCapability->timcheckstatus_param;
    break;
  case B_PARAM_TCPROP:
    mygeninfo = pCapability->timcheckprop_gen;
    myinfo = pCapability->timcheckprop_param;
    break;
  case B_PARAM_TCGENPROP:
    mygeninfo = pCapability->timcheckgenprop_gen;
    myinfo = pCapability->timcheckgenprop_param;
    break;
  case B_PARAM_MAGROUPINFO:
    mygeninfo = pCapability->mattrgroupinfo_gen;
    myinfo = pCapability->mattrgroupinfo_param;
    break;
  case B_PARAM_TAGROUPINFO:
    mygeninfo = pCapability->tattrgroupinfo_gen;
    myinfo = pCapability->tattrgroupinfo_param;
    break;
  case B_PARAM_CPUPORT:
    mygeninfo = pCapability->cpuport_gen;
    myinfo = pCapability->cpuport_param;
    break;
  case B_PARAM_STATICIO:
    mygeninfo = pCapability->staticio_gen;
    myinfo = pCapability->staticio_param;
    break;
  case B_PARAM_PUINFO:
    mygeninfo = pCapability->pu_gen;
    myinfo = pCapability->pu_param;
    break;
  case B_PARAM_BOARD:
    mygeninfo = pCapability->board_gen;
    myinfo = pCapability->board_param;
    break;
  case B_PARAM_SSTCFG:
    mygeninfo = pCapability->sstcfg_gen;
    myinfo = pCapability->sstcfg_param;
    break;
  case B_PARAM_SSTPROP:
    mygeninfo = pCapability->sst_gen;
    myinfo = pCapability->sst_param;
    break;
  case B_PARAM_CPCI_STATUS:
    mygeninfo = pCapability->cpcistatus_gen;
    myinfo = pCapability->cpcistatus_param;
    break;
  case B_PARAM_CPCI_PIN:
    mygeninfo = pCapability->cpcipin_gen;
    myinfo = pCapability->cpcipin_param;
    break;
  case B_PARAM_DATA_MEM:        /* no parameter info available */
  default:
    if (generate_errors == DO_STANDARD_ERROR_HANDLING)
      B_ERRETURN(B_E_DYNAMIC_CAPABILITY);
    else
      return B_E_DYNAMIC_CAPABILITY;
  }                             /*lint !e788 not all enums are used */

  /* check if the parameter is available at all */
  if (option == B_INDEX_SEARCH) /* just use the index to return the info */
  {
    if (parnum >= mygeninfo->num_elem)
    {
      if (generate_errors == DO_STANDARD_ERROR_HANDLING)
        B_ERRETURN(B_E_PARAM_NOT_EXIST);
      else
        return B_E_PARAM_NOT_EXIST;
    }
    id = parnum;                /* make the index point to parnum */
  }
  else if (option == B_ENUM_SEARCH)
  {
    /* search the parameter in the list using the enum as identifier */
    for (id = 0; id < mygeninfo->num_elem; id++)
    {
      if (parnum == myinfo[(int) id].proptyp.value)
        break;
    }
  }
  else if (option == B_NAME_SEARCH)
  {
    /* search the parameter using its name */
    for (id = 0; id < mygeninfo->num_elem; id++)
    {
      if (str_eq(myinfo[(int) id].param_name, (char *) parnum))
        break;
    }
  }
  else if (option == B_CLINAME_SEARCH)
  {
    /* search the parameter using its CLI name */
    for (id = 0; id < mygeninfo->num_elem; id++)
    {
      if (str_eq(myinfo[(int) id].param_short, (char *) parnum))
        break;
    }
  }
  else
  {
    /* TODO; proper response to this case */
    id = 0;
  }

  if (id == mygeninfo->num_elem)
  {
    if (generate_errors == DO_STANDARD_ERROR_HANDLING)
      B_ERRETURN(B_E_PARAM_NOT_EXIST);
    else
      return B_E_PARAM_NOT_EXIST;
  }

  if (paraminfo)                /* so it can get checked if the parameter is
                                 * there at all */
    *paraminfo = &myinfo[(int) id]; /* call with NULL just want to check that */

  if (generate_errors == DO_STANDARD_ERROR_HANDLING)
    B_ERRETURN(B_E_OK);

  return B_E_OK;
}

b_errtype EXPORT BestParamInfoGet(
    b_handletype handle,
    b_paramtype param_typ,      /* what kind of parameter at all */
    b_int64 parnum,             /* the parameter itself (as a number) */
    b_param_infotype ** paraminfo,  /* the info struct */
    b_int32 option              /* how to search for params (index or enum */
)
{
  B_ERRETURN(IBestParamInfoGet(handle, param_typ, parnum, paraminfo, option,
      DO_STANDARD_ERROR_HANDLING));
}
/* ------------------------------------------------------------------------
 * This function creates error strings out of the data given as parameters.
 * No further information is required. The error structure is not touched.
 * ------------------------------------------------------------------------ */
char *BestDynamicErrorStringGet(b_handletype handle,
    b_int64 group,
    b_int64 param,
    b_int64 val1,
    b_int64 val2,
    b_int64 val3,
    b_errtype theerr,
    char *buf)
{
  b_errtype err;

  b_param_infotype *myinfo = NULL;
  b_generic_infotype *mygeninfo;
  /* get generic info */
  err = IBestGenInfoGet(handle, (b_paramtype) group, &mygeninfo,
    DO_NO_ERROR_HANDLING);
  if (err != B_E_OK)
  {
    sprintf(buf, "Illegal parameter 'group' (%lu) for BestDynamicErrorStringGet"
      ,group);
    return buf;
  }

  /* get param info */
  if (theerr != B_E_PARAM_NOT_EXIST && theerr != B_EFW_PARAM_NOT_EXIST)
    err = IBestParamInfoGet(handle, (b_paramtype) group, param,
      &myinfo, (b_int32) B_ENUM_SEARCH, DO_NO_ERROR_HANDLING);
  if (err != B_E_OK)
  {
    sprintf(buf, "Illegal parameter 'param' (%lu) for BestDynamicErrorStringGet"
      ,param);
    return buf;
  }

  /* note; myinfo is guaranteed to be initialized from here on */

  switch (theerr)
  {
  case B_E_RANGE:
  case B_EFW_RANGE:
    sprintf(buf, "Parameter '%s' out of '%s' is out of range. "
      "Valid Range is from 0x%lx to 0x%lx. Value was 0x%lx",
      myinfo->param_name, mygeninfo->group_name, val1, val2, val3);
    break;
  case B_E_ALIGN:
  case B_EFW_ALIGN:
    sprintf(buf, "Alignment of Parameter '%s' out of '%s' is invalid. "
      "Value was 0x%lx",
      myinfo->param_name, mygeninfo->group_name, val2);
    break;
  case B_E_VALUE:
  case B_EFW_VALUE:
    sprintf(buf, "The Parameter '%s' out of '%s' is invalid. Value was 0x%lx",
      myinfo->param_name, mygeninfo->group_name, val1);
    break;
  case B_E_PARAM_NOT_EXIST:
  case B_EFW_PARAM_NOT_EXIST:
    sprintf(buf, "The Parameter '%ld' out of '%s' does not exist.",
      param, mygeninfo->group_name);
    break;
  default:
    sprintf(buf, "Illegal Errorvalue (%d) for BestDynamicErrorStringGet",
      (int) theerr);
    break;
  }                             /*lint !e788 */

  return buf;
}
/* ------------------------------------------------------------------------
 * This is a function to look for the valid license. In case the user
 * cannot do the requested task, it returns an error, otherwise it is ok.
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestCapabilityCheck(
    b_handletype handle,
    b_int32 capa_code
)
{
  b_errtype merr = B_E_NO_CAPABILITY;
  b_int32 counter;
  b_int32 cddp;
  b_int32 cddp2;
  HANDLECHECK;

  /* in case a trace memory license is asked, we also check for all bigger
   * values. Example: If you ask for 64K Tracemem but you do have a 1M
   * Tracemem depth license , it will give you an ok even that the 64K
   * Tracemem license is not explicitely on your board. And the license
   * checking for the tracememory depth is a special hack anyway. */
  if (capa_code & B_CAPABILITY_TRACEDEPTH_MASK)
  {
    merr = B_E_OK;

    cddp = capa_code & B_CAPABILITY_TRACEDEPTH_MASK;
    if (handle_array[handle].capable &&
      handle_array[handle].capable->capa_code)
      cddp2 = handle_array[handle].capable->capa_code[0] &
        B_CAPABILITY_TRACEDEPTH_MASK;
    else
      cddp2 = 0;

    if (cddp > cddp2)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1,
        (b_int32) B_CAPABILITY_TRACEDEPTH_SHIFT);
      merr = B_E_NO_CAPABILITY;
    }

    if ((merr == B_E_OK) && (capa_code & ~cddp))
    {                           /* something else is questioned here, call it
                                 * recursively */
      merr = BestCapabilityCheck(handle, capa_code & ~cddp);
    }
  }
  /* check for performance sequencer also */ 
  else if (capa_code & B_CAPABILITY_PERFSEQ_MASK)
  {
    merr = B_E_OK;

    cddp = capa_code & B_CAPABILITY_PERFSEQ_MASK;
    if (handle_array[handle].capable &&
      handle_array[handle].capable->capa_code)
      cddp2 = handle_array[handle].capable->capa_code[0] &
        B_CAPABILITY_PERFSEQ_MASK;
    else
      cddp2 = 0;

    if (cddp > cddp2)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1,
        (b_int32) B_CAPABILITY_PERFSEQ_SHIFT);
      merr = B_E_NO_CAPABILITY;
    }

    if ((merr == B_E_OK) && (capa_code & ~cddp))
    {                           /* something else is questioned here, call it
                                 * recursively */
      merr = BestCapabilityCheck(handle, capa_code & ~cddp);
    }
  }
  else
  {
    if (handle_array[handle].capable &&
      handle_array[handle].capable->capa_code &&
      (handle_array[handle].capable->capa_code[0] & capa_code) == capa_code)
    {
      merr = B_E_OK;
    }
    else
    {
      counter = 0;
      if (capa_code)            /* extract the information what license was
                                 * asked for */
      {
        while ((capa_code & 1UL) == 0)
        {
          counter++;
          capa_code >>= 1;
        }
      }
      BestLastErrorParamSet(handle, B_ERRPAR_1, counter);
      merr = B_E_NO_CAPABILITY;
    }
  }

  B_ERRETURN(merr);

}
/* ------------------------------------------------------------------------
 * This is a function to read back the complete license information
 * ------------------------------------------------------------------------ */
b_errtype EXPORT BestCapabilityRead(
    b_handletype handle,
    b_int32 * capa_code
)
{
  HANDLECHECK;

  if (handle_array[handle].capable)
    *capa_code = handle_array[handle].capable->capa_code[0];
  else
    *capa_code = B_CAPABILITY_NONE;

  B_ERRETURN(B_E_OK);
}
/* ------------------------------------------------------------------------
 * ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
 * void EXPORT BestParamValue2String ( const b_param_stringlisttype * thelist,
 *
 * Purpose	: get string value for num value if possible,
 *		  print num value otherwise
 *---------------------------------------------------------------------------*/
EXPORT const b_charptrtype BestParamValue2String (
  const b_param_stringlisttype * thelist,
  b_int32 value,
  b_charptrtype retStr
  )
{
  int i=0;
  assert (retStr);

  *retStr = '\0';

  if (thelist)
  {
    while (thelist[i].name && thelist[i].value != value)
    {
      i++;
    }
  }

  /* any of the following ptrs will be NULL if not found... */
  if (thelist && thelist[i].name && *(thelist[i].name))
  {
    strcpy (retStr, thelist[i].name);
  }
  else
  {
    sprintf (retStr, "%lu", value);
  }

  return retStr;
}

/*---------------------------------------------------------------------------*
 * void EXPORT BestParamValueListGet ( const b_param_infotype * pParamInfo,
 *
 * Purpose	: 
 *---------------------------------------------------------------------------*/
EXPORT void BestParamValueListGet ( const b_param_infotype * pParamInfo,
				    b_int32 minVal,
				    b_int32 maxVal,
				    b_charptrtype buffer )
{
  b_int32 i;			/* general purpose counter */
  b_int32 val;
  char numBuffer[11];
  b_int32 therule = pParamInfo->paramrule;
  b_int32 localMin, localMax;

  *buffer = '\0';		/* empty buffer */

  if (therule & B_PARAMRULE_BITSET) /* special handling for a bitset */
  {
    for (i = 0; i < 32UL; i++)
    {
      val = (1UL << i) + pParamInfo->min_val;
      
      if ( (i & pParamInfo->max_val) && (val >= minVal) )
      {
	if (*buffer)		/* already something in there? */
	{
	  strcat(buffer, ",");	/* add comma */
	}
	strcat ( buffer,
		 BestParamValue2String (pParamInfo->stringlist,
					val, numBuffer) );
      }
    }
  }
  else if (therule & B_PARAMRULE_ZERO2WIDTH)
  {
    localMax = pParamInfo->max_val == 32UL ?
      0xffffffffUL : (1UL << pParamInfo->max_val) - 1UL;
    localMax = min (maxVal, localMax);

    for (val = 0; val <= localMax; val++)
    {
      if (*buffer)		/* already something in there? */
      {
	strcat(buffer, ",");	/* add comma */
      }
      strcat ( buffer,
	       BestParamValue2String (pParamInfo->stringlist,
				      val, numBuffer) );
    }
  }
  else				/* the other rules basically are lists... */
  {
    b_int32 increment;
    
    /* do we need to add the value '0'? */
    if (therule & B_PARAMRULE_ZEROALLOWED)
    {
      if (minVal == 0)
      {
	if (*buffer)		/* already something in there? */
	{
	  strcat(buffer, ",");	/* add comma */
	}
	strcat (buffer,
		BestParamValue2String (pParamInfo->stringlist,
				       0UL, numBuffer));
      }
    }

    /* any specialities concerning bounds? */
    if (therule & B_PARAMRULE_128BOUND)
    {
      increment = 128UL;
    }
    if (therule & B_PARAMRULE_DWORDBOUND)
    {
      increment = 4UL;
    }
    if (therule & B_PARAMRULE_EVEN)
    {
      increment = 2UL;
    }
    else
    {
      increment = 1UL;
    }
    
    /* adjust min and max for specialities concerning boundaries */
    localMin = max (minVal, pParamInfo->min_val);
    localMax = min (maxVal, pParamInfo->max_val);

    if (localMin % increment != 0UL)
    {
      localMin = (localMin + increment) - (localMin % increment);
    }

    if (localMax % increment != 0UL)
    {
      localMax -= localMax % increment;
    }

    /* go fill the list */
    for (val=localMin; val<=localMax; val++)
    {
      if (*buffer)		/* already something in there? */
      {
	strcat(buffer, ",");	/* add comma */
      }
      strcat (buffer,
	      BestParamValue2String (pParamInfo->stringlist,
				     val, numBuffer));
    }

    /* do we need to add the value of MAXLONG? */
    if (therule & B_PARAMRULE_MAXLONGALLOWED)
    {
      if (maxVal == 0xFFFFFFFFUL)
      {
	if (*buffer)		/* already something in there? */
	{
	  strcat(buffer, ",");	/* add comma */
	}
	strcat (buffer,
		BestParamValue2String (pParamInfo->stringlist,
				       0xFFFFFFFF, numBuffer));
      }
    }
  }
}
